from rest_framework.response import Response
from rest_framework import status
import json
from rest_framework.exceptions import ParseError
from functools import wraps
from rest_framework.authtoken.models import Token
from app.entities.models import (User, Department, Company,
                                 ChangeRequestTemplateHeader,
                                 ChangeRequestFormHeader)
from app.entities import enums
from django.db.models import Q
from app.businesslayer.changerequest.change_request_template import (
    tmp_add_edit_delete,
    validation_approver_same_level,
    validation_existing_vendor_requestor,
    validation_existing_approver,
    validation_poc_vendor_only,
    validation_vendor_unique_level
)
from app.applicationlayer.utils import error_message


def error_safe(function):
    def wrap(request, *args, **kwargs):
        stat = status.HTTP_500_INTERNAL_SERVER_ERROR
        try:
            return function(request, *args, **kwargs)
        # except 'DoesNotExist' as ex:
        #     print('ddd')
        #     stat = status.HTTP_404_NOT_FOUND

        except Exception as ex:
            data = ''
            try:
                data = str(ex)
                print(f'ERROR: {data}')
            except Exception as ex2:
                pass
            return Response(data={"message": data},
                            status=stat)

    wrap.__doc__ = function.__doc__
    wrap.__name__ = function.__name__
    return wrap


class rms:

    # variables
    enums_super = enums.UserTypeEnum.SUPER_USER.value
    enums_OUA = enums.UserTypeEnum.OVERALL_USER_ADMIN.value
    enums_company = enums.UserTypeEnum.COMPANY_USER_ADMIN.value
    enums_department = enums.UserTypeEnum.DEPARTMENT_USER_ADMIN.value
    enums_user = enums.UserTypeEnum.USER.value
    access_error = "Logged user is unauthorize to access this section"
    department_error = 'Department should be same with the logged user'
    company_error = 'Company should be same with the logged user'

    def user(self):
        return self.request.user

    def user_type(self):
        return rms.user(self).user_type

    def superuser_create(self):
        if rms.user_type(self) != rms.enums_super:
            raise ParseError(rms.access_error)
        else:
            return True

    @staticmethod
    def user_create(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            print(rms.user_type(self))
            if rms.user_type(self) == rms.enums_user:
                # raise ParseError(
                #     rms.access_error
                # )
                return Response(
                    {"message": rms.access_error},
                    status=status.HTTP_400_BAD_REQUEST
                )
            elif rms.user_type(self) == rms.enums_department:
                if request.data['department'] != rms.user(self).department.code:
                    return Response(
                        {"message": rms.department_error},
                        status=status.HTTP_400_BAD_REQUEST
                    )
            elif rms.user_type(self) == rms.enums_company:
                user_company = rms.user(self).department.company
                request_department = Department.objects.filter(
                    Q(code=request.data['department']) &
                    Q(company=user_company)
                )
                if not request_department:
                    return Response(
                        {"message": rms.company_error},
                        status=status.HTTP_400_BAD_REQUEST
                    )
            return function(self, request, *args, **kwargs)
        return wrapper


    @staticmethod
    def ApplicationValidation(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            payload = request.data
            if ('excel_code' not in payload or
                    payload['excel_code'].strip() == ''
                ):
                return error_message(
                    '400', "Excel code is required",
                    'failed', status.HTTP_400_BAD_REQUEST
                )
            is_excel_code_exist = self.queryset.filter(
                excel_code=payload['excel_code']
            )
            if request.method.lower() == 'put':
                is_excel_code_exist = is_excel_code_exist.exclude(
                    code=self.kwargs['code']
                )
            if is_excel_code_exist:
                return error_message(
                    '400', "This excel code is already exists",
                    'failed', status.HTTP_400_BAD_REQUEST
                )
            return function(self, request, *args, **kwargs)
        return wrapper


    @staticmethod
    def AccountValidation(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            payload = request.data
            if 'username' not in payload or payload['username'].strip() == '':
                return error_message(
                    '400',
                    f"username is required",
                    'failed',
                    status.HTTP_400_BAD_REQUEST
                )
            username = self.queryset.filter(
                username=payload['username']
            )
            if request.method.lower() == 'put':
                username = username.exclude(code=self.kwargs['code'])
            if username.count() >= 1:
                return error_message(
                    '400',
                    f"username {username.first().username} is already taken",
                    'failed',
                    status.HTTP_400_BAD_REQUEST
                )
            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def reset_password(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            if rms.user_type(self) == rms.enums_user:
                raise ParseError(
                    rms.access_error
                )
            elif rms.user_type(self) == rms.enums_department:
                instance = self.get_object()
                if rms.user(self).department.code != instance.department.code:
                    raise ParseError(
                        rms.department_error
                    )
            elif rms.user_type(self) == rms.enums_company:
                user_company = rms.user(self).department.company
                request_department = Department.objects.filter(
                    Q(code=request.user.department.code) &
                    Q(company=user_company)
                )
                if not request_department:
                    raise ParseError(rms.company_error)

            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def company_crate(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            rms.superuser_create(self)
            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def department_crate(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            rms.superuser_create(self)
            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def application_crate(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            rms.superuser_create(self)
            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def module_crate(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):
            rms.superuser_create(self)
            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def company_list(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):

            if rms.user_type(self) == rms.enums_company or rms.user_type(self) == rms.enums_department:

                id = rms.user(self).department.company.id

                self.queryset = self.queryset.filter(
                    id=id
                )
            elif rms.user_type(self) == rms.enums_super:
                pass
            elif rms.user_type(self) == rms.enums_OUA:
                pass
            else:
                raise ParseError(
                    rms.access_error
                )

            return function(self, request, *args, **kwargs)
        return wrapper

    @staticmethod
    def department_list(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):

            if rms.user_type(self) == rms.enums_department:
                id = rms.user(self).department.id

                self.queryset = self.queryset.filter(
                    id=id
                )
            elif rms.user_type(self) == rms.enums_OUA:
                pass
            elif rms.user_type(self) == rms.enums_company:
                pass
            elif rms.user_type(self) == rms.enums_super:
                pass
            else:
                raise ParseError(
                    rms.access_error
                )

            return function(self, request, *args, **kwargs)
        return wrapper


    @staticmethod
    def user_list(function):
        @wraps(function)
        def wrapper(self, request, *args, **kwargs):

            if rms.user_type(self) == rms.enums_department:
                code = rms.user(self).department.code

                self.queryset = self.queryset.filter(
                    department=str(code)
                )

            elif rms.user_type(self) == rms.enums_company:
                code = rms.user(self).department.company.code
                self.queryset = self.queryset.filter(
                    department__company=str(code)
                )
            elif rms.user_type(self) == rms.enums_super:
                pass
            elif rms.user_type(self) == rms.enums_OUA:
                pass
            else:
                raise ParseError(
                    rms.access_error
                )

            return function(self, request, *args, **kwargs)
        return wrapper


def TemplateValidation(function):
    @wraps(function)
    def wrapper(self, request, *args, **kwargs):
        template_header = request.data

        required = {'requested_to_template_name': 'Template Name',
                    'requested_to_template_id': 'CR Number prefix',
                    'requested_to_target_date': 'Lead Time',
                    'requested_to_company': 'Company',
                    'requested_to_department': 'Department'}
        
        for key in required.keys():
            if (not key in template_header or
                    str(template_header[key]).strip() == ''):
                return error_message('400', required[key] + ' is required',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        # Restrict form using Superuser Department
        if (template_header['created_by_department'] == 'DEPARTMENT-20190923-0000001' or
                template_header['requested_to_department'] == 'DEPARTMENT-20190923-0000001'):
            return error_message('400', 'Superuser department cannot be selected',
                    'failed', status.HTTP_400_BAD_REQUEST)
            
        tmp_approvers = template_header['tmp_approvers']
        
        # Check if Vendor and Requestor are existing on routing table
        if len(tmp_approvers) < 2:
            return error_message('400', 'Please make sure to add an Approver, Vendor and Requestor into routing table',
                'failed', status.HTTP_400_BAD_REQUEST)
        else:
            result = validation_existing_vendor_requestor(tmp_approvers)

            if result is False:
                return error_message('400', 'Please add Vendor/Implementor and Requestor into routing table',
                    'failed', status.HTTP_400_BAD_REQUEST)

        # Do not allow adding an approver for the same level
        validation_result = validation_approver_same_level(tmp_approvers)
        if validation_result is not None:
            return error_message('400', validation_result + ' is already exist for the same level of approval.',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        # Do not allow saving user as Vendor and other delegation
        validate = validation_poc_vendor_only(
            template_header['requested_to_user'], tmp_approvers)
        if validate is True:
            return error_message('400', 'Point of contact can only be assign to Vendor/Implementor',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        # Do not allow to save Vendor same level to other delegation
        validate = validation_vendor_unique_level(tmp_approvers)
        if not validate == 0:
            return error_message('400', 'Level ' + str(validate) + ' is already assigned for the Vendor/Implementor.',
                    'failed', status.HTTP_400_BAD_REQUEST)
            
        return function(self, request, *args, **kwargs)
    return wrapper


def FormValidation(function):
    @wraps(function)
    def wrapper(self, request, *args, **kwargs):
        form_header = request.data
        
        required = {'requested_to_template_name': 'CR Name',
                    'requested_to_target_date': 'Date Required',
                    'requested_to_company': 'Company',
                    'requested_to_department': 'Department',
                    'requested_to_user': 'Point of Contact',
                    'requested_to_priority': 'Priority Level'}
        
        for key in required.keys():
            if not key in form_header or str(form_header[key]).strip() == '':
                return error_message('400', required[key] + ' is required',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        # restrict superuser department
        if (form_header['requested_by_department'] == 'DEPARTMENT-20190923-0000001' or
                form_header['requested_to_department'] == 'DEPARTMENT-20190923-0000001'):
            return error_message('400', 'Superuser department cannot be selected',
                'failed', status.HTTP_400_BAD_REQUEST)
        
        frm_approvers = form_header['frm_approvers']
        
        # Check if Vendor and Requestor are existing on routing table
        if len(frm_approvers) < 2:

            return error_message('400', 'Please make sure to add an Approver, Vendor and Requestor into routing table',
                'failed', status.HTTP_400_BAD_REQUEST)
        else:
            result = validation_existing_vendor_requestor(frm_approvers)

            if result is False:

                return error_message('400', 'Please add Vendor/Implementor and Requestor into routing table',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        # Do not allow adding an approver for the same level
        validation_result = validation_approver_same_level(frm_approvers)
        
        if validation_result is not None:
            return error_message('400', validation_result + ' is already exist for the same level of approval.',
                'failed', status.HTTP_400_BAD_REQUEST)
        
        # Restrict a requestor to submit a CR without atleast 1 approver
        counter = validation_existing_approver(frm_approvers)
        if form_header['status'].lower() == 'pending' and counter == 0:
            return error_message('400', 'Please select at least 1 approver before submitting this request.',
                'failed', status.HTTP_400_BAD_REQUEST)
        
        # Do not allow saving user as Vendor and other delegation
        validate = validation_poc_vendor_only(
            form_header['requested_to_user'], frm_approvers)
        if validate is True:
            return error_message('400', 'Point of contact can only be assign to Vendor/Implementor',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        # Do not allow to save Vendor same level to other delegation
        validate = validation_vendor_unique_level(frm_approvers)
        if validate is True:
            return error_message('400', 'Vendor/Implementor cannot have same level with other delegation/s',
                    'failed', status.HTTP_400_BAD_REQUEST)
        
        return function(self, request, *args, **kwargs)
    return wrapper
