from datetime import datetime
from django.db.models.functions import Lower
from functools import wraps
from django.conf import settings
import requests
from app.entities import models
from datetime import timedelta
from django.db.models import Q
from app.applicationlayer.utils import main_threading, notification_create
from app.helper.email_service import sender
from app.applicationlayer.utils import log_save
from app.entities import enums
from django.forms.models import model_to_dict
from rest_framework.exceptions import ParseError
from django.db import IntegrityError
from rest_framework.exceptions import APIException 
from django.db.models import Max


CR_FRONT_LINK = settings.CR_FRONT_LINK
APPROVER_MESSAGE = settings.APPROVER_MESSAGE
REQUESTOR_MESSAGE = settings.REQUESTOR_MESSAGE
REQUESTOR_REJECT_MESSAGE = settings.REQUESTOR_REJECT_MESSAGE
VENDOR_ACKNOWLEDGE_MESSAGE = settings.VENDOR_ACKNOWLEDGE_MESSAGE
REQUESTOR_ACKNOWLEDGE_MESSAGE = settings.REQUESTOR_ACKNOWLEDGE_MESSAGE
REQUESTOR_COMPLETION_MESSAGE = settings.REQUESTOR_COMPLETION_MESSAGE
VENDOR_ACCEPTANCE_MESSAGE = settings.VENDOR_ACCEPTANCE_MESSAGE
VENDOR_REJECT_MESSAGE = settings.VENDOR_REJECT_MESSAGE


def entity_log_bulk(queryset, entity, tbl):
    try:
        # print(entity)
        for data in queryset:
            test = model_to_dict(data)
            log_save(
                enums.LogEnum.DELETED.value,
                entity,
                test['id'],
                test,
                ''
            )
            tbl.objects.filter(id=test['id']).delete()
        return True
    except IntegrityError as exc:
        raise APIException(detail=exc)


def get_dept_details(dept_no):
    dept_instance = models.Department.objects.filter(code=dept_no)

    return dept_instance


def get_companies_details(slug):
    company_instance = models.Company.objects.filter(code=slug)

    return company_instance


def get_account_details(id_number):
    account_instance = models.User.objects.filter(code=id_number)
    return account_instance


def get_allowed_company(id_number):
    return requests.get(f'{ALLOWED_COMPANY}?id_number={id_number}')


class QuerySetHelper:

    @staticmethod
    def Sort(context):
        sort_field = context.request.query_params.get('sort_field')
        sort_order = context.request.query_params.get('sort_order')
        if sort_field and sort_order:
            if sort_order.lower() == 'asc':
                context.queryset = context.queryset.order_by(sort_field)
            else:
                context.queryset = context.queryset.order_by(f"-{sort_field}")
        return context.queryset

    @staticmethod
    def Search(context):
        search_field = context.request.query_params.get('search-field')
        search_key = context.request.query_params.get('search-key')
        if search_field and search_key:
            context.queryset = context.queryset(
                __raw__={f"{search_field}" : {"$regex" : f".*{search_key.lower()}.*"}}
            )
        return context.queryset
    
    @staticmethod
    def SearchDynamic(base_queryset,
                      ):
        search_field = context.request.query_params.get('search-field')
        search_key = context.request.query_params.get('search-key')
        if search_field and search_key:
            context.queryset = context.queryset(
                __raw__={f"{search_field}" : {"$regex" : f".*{search_key.lower()}.*"}}
            )
        return context.queryset

    @staticmethod
    def Filter(context):
        if int(len(context.request.query_params)) > 0:
            filtering_kwargs = {}
            with_params = []
            common_params = (
                'page', 'page-size', 'page_size', 'sort_order', 'sort_field'
            )
            for field, value in context.request.GET.items():
                filtering_kwargs = {}
                if value and field.lower() not in common_params:
                    filtering_kwargs[field] = {"$regex" : f".*{value.lower()}.*"}
                    filtering_kwargs[field] = {"$regex" : f".*{value}.*"}
                    # filtering_kwargs[field] = {"$regex" : f".*{value.lower()}.*"}
                    with_params.append(filtering_kwargs)
                    raw_query = {"$or": with_params}
                    context.queryset = context.queryset(__raw__=raw_query)
        return context.queryset


def ApproverStatus(status):
    choices = ["pending", "rejected", "approved", "completed", "cancelled",
               'acknowledged', 'accepted']
    if status not in choices:
        return False
    else:
        return True


def number_generator(prefix, id):
    date = '{:%Y%m%d}'.format(datetime.now())
    id_num = '{:07}'.format(id)
    autogenerated_no = prefix + '-' + date + '-' + id_num

    return autogenerated_no


def get_template_instance(form_code):
    template_instance = models.ChangeRequestFormHeader.objects.filter(
        Q(form_code=form_code)
    ).first()
    return template_instance


def send_mail_vendor(requestor,
                     form_code,
                     delegation,
                     msg,
                     action,
                     remarks,
                     routing_level):

    cr_link = f'{CR_FRONT_LINK}/{form_code}'
    template_instance = get_template_instance(form_code)

    cr_number = template_instance.requested_to_template_id
    template_name = template_instance.requested_to_template_name
    requested_to_company = template_instance.requested_to_company.code
    requested_to_department = template_instance.requested_to_department.code
    requested_by_user = template_instance.requested_by_user.code
    requested_to_priority = template_instance.requested_to_priority
    status = template_instance.status
    requested_to_user = template_instance.requested_to_user.code
    
    # receiver details
    vendor_instance = get_account_details(requested_to_user)
    vendor_name = vendor_instance.values_list('name', flat=True)[0]
    vendor_email = vendor_instance.values_list('email', flat=True)[0]
    vendor_code = vendor_instance.values_list('code', flat=True)[0]
    
    # sender details
    sender_instance = get_account_details(requestor)
    sender_name = sender_instance.values_list('name', flat=True)[0]
    sender_code = sender_instance.values_list('code', flat=True)[0]
    sender_email = sender_instance.values_list('email', flat=True)[0]
    
    # department details
    department = get_dept_details(requested_to_department)
    dept_name = department.values_list('name', flat=True)[0]

    company = get_companies_details(requested_to_company)
    company_name = company.values_list('name', flat=True)[0]

    # call sender email

    name = vendor_name
    action_by = sender_name
    routing_level = routing_level
    status = status
    cr_number = cr_number
    cr_name = template_name
    company_requestedto = company_name
    department_requestedto = dept_name
    priority_level = requested_to_priority
    url = cr_link
    remarks = remarks

    recipient = vendor_email
    action_type = action
    delegation_type = delegation
    admin = sender_email

    args = [name, action_by, routing_level, status, cr_number, cr_name,
            company_requestedto, department_requestedto, priority_level,
            url, remarks, recipient, action_type, delegation_type, admin]

    main_threading(args, sender.routing_table_actions)

    message = f"{sender_name} {msg} ({template_name})"

    # create notification

    notification_create(form_code, message, vendor_code,
                        sender_code)


def send_mail_requestor(current_user,
                        form_code,
                        delegation,
                        msg,
                        action,
                        remarks,
                        routing_level):

    cr_link = f'{CR_FRONT_LINK}/{form_code}'
    template_instance = get_template_instance(form_code)
    
    cr_number = template_instance.requested_to_template_id
    template_name = template_instance.requested_to_template_name
    requested_to_company = template_instance.requested_to_company.code
    requested_to_department = template_instance.requested_to_department.code
    requested_by_user = template_instance.requested_by_user.code
    requested_to_priority = template_instance.requested_to_priority
    
    # receiver details  --------------------------------------------------

    requestor_instance = get_account_details(requested_by_user)
    requestor_name = requestor_instance.values_list('name', flat=True)[0]
    requestor_email = requestor_instance.values_list('email', flat=True)[0]
    requestor_code = requestor_instance.values_list('code', flat=True)[0]

    # sender details    --------------------------------------------------
    sender_instance = get_account_details(current_user)
    sender_email = sender_instance.values_list('email', flat=True)[0]
    sender_name = sender_instance.values_list('name', flat=True)[0]

    department = get_dept_details(requested_to_department)
    dept_name = department.values_list('name', flat=True)[0]

    company = get_companies_details(requested_to_company)
    company_name = company.values_list('name', flat=True)[0]

    # call sender email

    name = requestor_name
    action_by = sender_name
    routing_level = routing_level
    status = action
    cr_number = cr_number
    cr_name = template_name
    company_requestedto = company_name
    department_requestedto = dept_name
    priority_level = requested_to_priority
    url = cr_link
    
    recipient = requestor_email
    action_type = action
    delegation_type = delegation
    admin = sender_email

    args = [name, action_by, routing_level, status, cr_number, cr_name,
            company_requestedto, department_requestedto, priority_level,
            url, remarks, recipient, action_type, delegation_type, admin]

    main_threading(args, sender.routing_table_actions)

    message = f"{sender_name} {msg} ({template_name})"

    # create notification

    notification_create(form_code, message, requestor_code,
                        current_user)


def next_approver_email(form_code, next_level):
    cr_link = f'{CR_FRONT_LINK}/{form_code}'
    template_instance = get_template_instance(form_code)
    
    cr_number = template_instance.requested_to_template_id
    template_name = template_instance.requested_to_template_name
    requested_to_company = template_instance.requested_to_company.code
    requested_to_department = template_instance.requested_to_department.code
    requested_by_user = template_instance.requested_by_user.code
    requested_to_priority = template_instance.requested_to_priority
    cr_status = template_instance.status

    # requestor details  --------------------------------------------------
    sender_instance = get_account_details(requested_by_user)
    sender_email = sender_instance.values_list('email', flat=True)[0]
    sender_name = sender_instance.values_list('name', flat=True)[0]
    sender_code = sender_instance.values_list('code', flat=True)[0]

    department = get_dept_details(requested_to_department)
    dept_name = department.values_list('name', flat=True)[0]

    company = get_companies_details(requested_to_company)
    company_name = company.values_list('name', flat=True)[0]

    # get details of next approver/s
    next_approver = models.ChangeRequestFormApprovers.objects.filter(
                                                level=str(next_level),
                                                form_code=form_code
                                                )
    
    # LOOP on next approver for sending email
    for n_approver in next_approver:
        
        # NOTIF MSG FOR NEXT APPROVER
        msg = APPROVER_MESSAGE.split(';')[0]
        
        if n_approver.delegation.lower() == 'vendor/implementor':
            msg = VENDOR_ACKNOWLEDGE_MESSAGE.split(';')[0]

        # next approver details  --------------------------------------------------
        receiver_instance = get_account_details(n_approver.user.code)
        receiver_name = receiver_instance.values_list('name', flat=True)[0]
        receiver_email = receiver_instance.values_list('email', flat=True)[0]
        receiver_code = receiver_instance.values_list('code', flat=True)[0]

        # call sender email

        name = receiver_name
        cr_number = cr_number
        cr_name = template_name
        company_requestedto = company_name
        department_requestedto = dept_name
        priority_level = requested_to_priority
        status = cr_status
        url = cr_link
        
        recipient = receiver_email
        delegation_type = n_approver.delegation.lower()
        admin = sender_email

        args = [name, cr_number, cr_name,
                company_requestedto, department_requestedto, priority_level,
                status, url, recipient, delegation_type, admin]

        main_threading(args, sender.routing_table_actions_required)

        message = f"{sender_name} {msg} ({template_name})"

        # create notification

        notification_create(form_code, message, receiver_code,
                            sender_code)


def cancel_overdue(request):
    date_submitted = datetime.now()
    requestor = request.data['requested_by_user']
    requestor = requests.get(f'{ACCOUNTS}{requestor}/')
    requestor = requestor.json()['results']

    cancel_date = date_submitted + timedelta(days=30)
    cancel_date = cancel_date.strftime('%Y-%m-%d 00:00:00.000')

    request.data['date_submitted'] = date_submitted
    request.data['cancel_date'] = cancel_date

    email_content_cancel = {
        "sender_account_no": requestor['id_number'],
        "receiver_account_no": requestor['id_number'],
        "email_code": "RMS-CRCANCELLED",
        "email_recipient": requestor['email'],
        "app": "CMS",
        "sent": "False",
        "name": requestor['name'],
        "status": "Pending",
        "auto_cancel_date": cancel_date,
        "cr_number": request.data['requested_to_template_id'],
        "cr_name": request.data['requested_to_template_name'],
        "company_requestedto": request.data['requested_to_company'],
        "department_requestedto": request.data['requested_to_department'],
        "priority_level": request.data['requested_to_priority'],
        "url": "http://devweb.rms.oneberrysystem.com/login"
    }

    exist_cancel_template = models.CancelDateCR.objects.filter(
        cr_number=request.data['requested_to_template_id']
    )
    if exist_cancel_template:
        exist_cancel_template.delete()

    models.CancelDateCR.objects.create(
        cr_number=request.data['requested_to_template_id'],
        trigger_date=cancel_date,
        email_content=email_content_cancel
    )

    requested_to_target_date = parser.parse(
        request.data['requested_to_target_date']
    )

    email_content_cancel['email_code'] = "RMS-CROVERDUE"
    email_content_cancel['target_date'] = requested_to_target_date

    overdue = requested_to_target_date + timedelta(days=30)
    overdue = overdue.strftime('%Y-%m-%d 00:00:00.000')

    models.TargetDateOverdue.objects.create(
        cr_number=form_code,
        trigger_date=overdue,
        email_content=email_content_cancel
    )
    return True


def get_max_batchno(request):
    try:
        max_id = models.ChangeRequestHistory.objects.all(
            ).order_by("-batch_no")[0]

        if max_id:
            max_batch = max_id.batch_no.split("-")
            max_batch = int(max_batch[2].lstrip("0")) + 1
            batchno = number_generator("BATCH", max_batch)
        else:
            batchno = number_generator("BATCH", 1)
    except:
        batchno = number_generator("BATCH", 1)

    return batchno


def crhistory_save(batch_no, main_action,
                   action, entity, form_code,
                   fromValue, toValue):

    models.ChangeRequestHistory.objects.create(
            batch_no=batch_no,
            main_action=main_action,
            action=action,
            entity=entity,
            form_code=form_code,
            fromValue=fromValue,
            toValue=toValue
        )
    
    return True


def crhistory_log_bulk_delete(queryset, entity, tbl, form_code,
                              batch_no, main_action):
    try:
        # print(entity)
        for data in queryset:
            test = model_to_dict(data)
            crhistory_save(batch_no,
                           main_action,
                           enums.CREnum.DELETED.value,
                           entity,
                           form_code,
                           test,
                           '')
            
            tbl.objects.filter(id=test['id']).delete()
        return True
    except IntegrityError as exc:
        raise APIException(detail=exc)


def generate_template_id(prefix):
    print("hello")