from django.shortcuts import render

from rest_framework import viewsets as meviewsets
from rest_framework.views import APIView
from app.applicationlayer.management.changerequest import serializers
from app.entities import models
from app.applicationlayer import paginators

from datetime import datetime
from datetime import timedelta

from rest_framework.response import Response
from rest_framework import status, views

from rest_framework.decorators import action
from django.http import Http404

from django.db.models import Q
import requests
from django.conf import settings
from rest_framework.exceptions import ValidationError
from django.db import transaction, IntegrityError, connection
from app.applicationlayer.utils import QuerySetHelper
from app.businesslayer.changerequest import change_request


APPROVER_MESSAGE = settings.APPROVER_MESSAGE
REQUESTOR_MESSAGE = settings.REQUESTOR_MESSAGE
REQUESTOR_REJECT_MESSAGE = settings.REQUESTOR_REJECT_MESSAGE
VENDOR_ACKNOWLEDGE_MESSAGE = settings.VENDOR_ACKNOWLEDGE_MESSAGE
REQUESTOR_ACKNOWLEDGE_MESSAGE = settings.REQUESTOR_ACKNOWLEDGE_MESSAGE
REQUESTOR_COMPLETION_MESSAGE = settings.REQUESTOR_COMPLETION_MESSAGE
VENDOR_ACCEPTANCE_MESSAGE = settings.VENDOR_ACCEPTANCE_MESSAGE
VENDOR_REJECT_MESSAGE = settings.VENDOR_REJECT_MESSAGE


class ChangeRequestTemplatesViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestTemplateHeader.objects.all()
    lookup_field = 'template_no'
    serializer_class = serializers.ChangeRequestTemplatesSerializer
    pagination_class = paginators.SimplePageNumberPagination

    def list(self, request, *args, **kwargs):
        
        self.queryset = self.queryset.filter(
            deleted_at=None
        )

        self.queryset = self.queryset.order_by('-created')
        self.queryset = QuerySetHelper.Sort(self)

        return super(ChangeRequestTemplatesViewset, self).list(request)

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(
            deleted_at=None,
            template_no=str(self.kwargs.get('template_no'))
        ).first()

        serializer = self.get_serializer(instance)
        return Response(serializer.data)

    def destroy(self, request, *args, **kwargs):
        try:
            template_no = self.kwargs['template_no']
            instance = models.ChangeRequestTemplateHeader.objects.filter(
                Q(template_no=template_no) &
                Q(deleted_at=None)
            ).update(deleted_at=datetime.now())
            return Response({"message": "Deleted"}, status=status.HTTP_200_OK)
        except Exception as e:
            return Response(e,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    def partial_update(self, request, *args, **kwargs):
        try:
            kwargs['partial'] = True
            template_no = self.kwargs['template_no']
            instance = models.ChangeRequestTemplateHeader.objects.filter(
                template_no=template_no
            ).order_by('-deleted_at')
            instance.update(deleted_at=None)
            message = "Archived Change Request restored"
            return Response(message, status=status.HTTP_200_OK)

        except Exception as e:
            message = {
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                      }
            return Response(message, 
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class ChangeRequestTemplateApproversViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestTemplateApprovers.objects.all()
    serializer_class = serializers.ChangeRequestTemplateApproversSerializer
    pagination_class = paginators.SimplePageNumberPagination
    lookup_field = "code"

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        
        return super(ChangeRequestTemplateApproversViewset, self).list(request)

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestTemplateAttachmentsViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestTemplateAttachments.objects.all()
    serializer_class = serializers.ChangeRequestTemplateAttachmentsSerializer
    pagination_class = paginators.SimplePageNumberPagination
    lookup_field = "code"

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)

        return super(
            ChangeRequestTemplateAttachmentsViewset, self
        ).list(request)


class ChangeRequestTemplateStakeHoldersViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestTemplateStakeHolders.objects.all()
    serializer_class = serializers.ChangeRequestTemplateStakeHoldersSerializer
    pagination_class = paginators.SimplePageNumberPagination
    lookup_field = "code"

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        
        return super(ChangeRequestTemplateStakeHoldersViewset, self).list(request)

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestTemplateDetailsViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestTemplateDetails.objects.all()
    serializer_class = serializers.ChangeRequestTemplateDetailsSerializer
    pagination_class = paginators.SimplePageNumberPagination
    lookup_field = "code"

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        
        return super(ChangeRequestTemplateDetailsViewset, self).list(request)

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestTemplatePost(APIView):

    @transaction.atomic()
    def post(self, request):
        template_header = request.data

        try:
            data_list_approver = []
            data_list_stake = []
            data_list_attach = []
            data_list_detail = []

            template_header_data = {
                'requested_to_template_name': template_header['requested_to_template_name'],
                'requested_to_template_id': template_header['requested_to_template_id'],
                'requested_to_objective': template_header['requested_to_objective'],
                'requested_to_target_date': template_header['requested_to_target_date'],
                'requested_to_priority': template_header['requested_to_priority'],
                'description': template_header['description'],
                'created_by_department': template_header['created_by_department'],
                'created_by_user': template_header['created_by_user'],
                'requested_to_company': template_header['requested_to_company'],
                'requested_to_department': template_header['requested_to_department'],
                'requested_to_user': template_header['requested_to_user']
            }

            tmp_approvers = template_header['tmp_approvers']
            tmp_stakes = template_header['tmp_stakes']
            tmp_attachments = template_header['tmp_attachments']
            tmp_details = template_header['tmp_details']

            sp1 = transaction.savepoint()  # nothing will save to db

            serializer = serializers.ChangeRequestTemplatesSerializer(
                data=template_header_data)
            
            if serializer.is_valid(raise_exception=True):
                serializer.save()
            
            tmp_id = serializer.data['template_no']

            # create template approvers

            for tmp_approver in tmp_approvers:
                tmp_approver['template_no'] = tmp_id
                data_list_approver.append(tmp_approver)

            serializerApprover = serializers.ChangeRequestTemplateApproversSerializer(
                data=data_list_approver, many=True)

            if serializerApprover.is_valid(raise_exception=True):
                serializerApprover.save()

            # create template stakes

            for tmp_stake in tmp_stakes:
                tmp_stake['template_no'] = tmp_id
                data_list_stake.append(tmp_stake)

            serializerStake = serializers.ChangeRequestTemplateStakeHoldersSerializer(
                data=data_list_stake, many=True)

            if serializerStake.is_valid(raise_exception=True):
                serializerStake.save()
            
            # create template attachments

            for tmp_attachment in tmp_attachments:
                tmp_attachment['template_no'] = tmp_id
                data_list_attach.append(tmp_attachment)

            serializerAttach = serializers.ChangeRequestTemplateAttachmentsSerializer(
                data=data_list_attach, many=True)

            if serializerAttach.is_valid(raise_exception=True):
                serializerAttach.save()

            # create template details
            
            for tmp_detail in tmp_details:
                tmp_detail['template_no'] = tmp_id
                data_list_detail.append(tmp_detail)

            serializerDetail = serializers.ChangeRequestTemplateDetailsSerializer(
                data=data_list_detail, many=True)
            
            if serializerDetail.is_valid(raise_exception=True):
                serializerDetail.save()

            message = {
                'code': 201,
                'status': 'success',
                'message': 'Template Details successfully saved!',
                'results': serializer.data
            }

            return Response(message, status=status.HTTP_201_CREATED)

        except ValidationError as e:
            transaction.savepoint_rollback(sp1)
            message = {
                'code': 400,
                'status': 'failed',
                'message': str(e),
            }
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            transaction.savepoint_rollback(sp1)
            message = {
                'code': 500,
                'status': 'failed',
                'message': 'Request was not able to process' + str(e),
            }
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)


# Change Request Form Views

class ChangeRequestFormsViewset(meviewsets.ModelViewSet):

    serializer_class = serializers.ChangeRequestFormHeaderSerializer
    pagination_class = paginators.SimplePageNumberPagination
    queryset = models.ChangeRequestFormHeader.objects.all()
    lookup_field = 'form_code'

    def list(self, request, *args, **kwargs):
        id_number = self.request.user
        
        # print(id_number.code)
        self.queryset = change_request.list_by_user(id_number.code)
        
        self.queryset = change_request.filter_base(
            self.queryset,
            request.query_params.get('company_requested_to'),
            request.query_params.get('department_requested_to'),
            request.query_params.get('date_modified_from'),
            request.query_params.get('date_modified_to'),
            request.query_params.get('date_required_from'),
            request.query_params.get('date_required_to'),
            request.query_params.get('form_type'),
        )

        if self.request.query_params.get('search'):
            search_key = self.request.query_params.get('search')
        
            self.queryset = self.queryset.filter(
                Q(requested_to_template_name__icontains=search_key.lower()) |
                Q(requested_to_template_id__icontains=search_key.lower())
            )
        
        self.queryset = self.queryset.order_by('-created')
        self.queryset = QuerySetHelper.Sort(self)
        
        return super(ChangeRequestFormsViewset, self).list(request)

    @action(detail=False,
            methods=['get'],
            url_path='dashboard',
            name="Dashboard Summary")
    def dashboard_view(self, request):

        id_number = self.request.user

        # id_number = 'Acct-20190813-0000138'
        
        self.queryset = change_request.list_by_user(id_number.code)

        self.queryset = change_request.filter_base(
            self.queryset,
            request.query_params.get('company_requested_to'),
            request.query_params.get('department_requested_to'),
            request.query_params.get('date_modified_from'),
            request.query_params.get('date_modified_to'),
            request.query_params.get('date_required_from'),
            request.query_params.get('date_required_to'),
            request.query_params.get('form_type'),
        )

        pending = self.queryset.filter(
            status__iexact='Pending'
        ).count()

        approved = self.queryset.filter(
            status__iexact='Approved'
        ).count()

        rejected = self.queryset.filter(
            status__iexact='Rejected'
        ).count()

        cancelled = self.queryset.filter(
            status__iexact='Cancelled'
        ).count()

        completed = self.queryset.filter(
            status__iexact='Closed'  #Completed
        ).count()

        high = self.queryset.filter(
            requested_to_priority='High'
        ).count()

        normal = self.queryset.filter(
            requested_to_priority='Normal'
        ).count()

        awaiting_filtered = change_request.filter_awaiting(self.queryset, id_number)

        # awaiting = awaiting_filtered.count()
        awaiting = 0

        for awaits in awaiting_filtered:

            if awaits.status.lower() == 'rejected':
                print("rejected")
            elif awaits.status.lower() == 'closed':
                print("closed")
            elif awaits.status.lower() == 'cancelled':
                print("cancelled")
            else:
                awaiting = awaiting + 1
        
        overdue_filtered = change_request.filter_overdue(self.queryset)

        overdue = overdue_filtered.count()
        
        message = {
            'account_no': id_number,
            'pending': pending,
            'approved': approved,
            'rejected': rejected,
            'cancelled': cancelled,
            'completed': completed,
            'high': high,
            'normal': normal,
            'awaiting': awaiting,
            'overdue': overdue,
            'code': 200,
            'status': 'success',
            'message': 'Dashboard Summary'
        }
       
        return Response(message, status=status.HTTP_200_OK)
        

    @action(detail=False,
            methods=['get'],
            url_path='status',
            name="Dashboard Summary Status")
    def list_by_status_view(self, request):

        id_number = self.request.user

        # id_number = 'Acct-20190531-0000002'

        self.queryset = change_request.list_by_user(id_number.code)

        self.queryset = change_request.filter_status(
            self.queryset,
            request.query_params.get('status')
        )

        self.queryset = change_request.filter_base(
            self.queryset,
            request.query_params.get('company_requested_to'),
            request.query_params.get('department_requested_to'),
            request.query_params.get('date_modified_from'),
            request.query_params.get('date_modified_to'),
            request.query_params.get('date_required_from'),
            request.query_params.get('date_required_to'),
            request.query_params.get('form_type'),
        )
        
        self.queryset = self.queryset.order_by('-created')
        self.queryset = QuerySetHelper.Sort(self)
        # self.queryset = QuerySetHelper.Filter(self)
        return super(ChangeRequestFormsViewset, self).list(request)
    
    
    @action(detail=False,
            methods=['get'],
            url_path='overdue',
            name="Dashboard Summary Overdue")
    def list_by_overdue_view(self, request):

        id_number = self.request.user

        # id_number = 'Acct-20190531-0000002'

        self.queryset = change_request.list_by_user(id_number.code)

        self.queryset = change_request.filter_overdue(self.queryset)

        self.queryset = change_request.filter_base(
            self.queryset,
            request.query_params.get('company_requested_to'),
            request.query_params.get('department_requested_to'),
            request.query_params.get('date_modified_from'),
            request.query_params.get('date_modified_to'),
            request.query_params.get('date_required_from'),
            request.query_params.get('date_required_to'),
            request.query_params.get('form_type'),
        )

        self.queryset = self.queryset.order_by('-created')
        self.queryset = QuerySetHelper.Sort(self)
        return super(ChangeRequestFormsViewset, self).list(request)


    @action(detail=False,
            methods=['get'],
            url_path='awaiting',
            name="Dashboard Summary Awaiting")
    def list_by_awaiting_view(self, request):

        id_number = self.request.user

        # id_number = 'Acct-20190813-0000137'

        self.queryset = change_request.list_by_user(id_number.code)

        self.queryset = change_request.filter_awaiting(self.queryset, id_number)

        self.queryset = change_request.filter_base(
            self.queryset,
            request.query_params.get('company_requested_to'),
            request.query_params.get('department_requested_to'),
            request.query_params.get('date_modified_from'),
            request.query_params.get('date_modified_to'),
            request.query_params.get('date_required_from'),
            request.query_params.get('date_required_to'),
            request.query_params.get('form_type'),
        )

        self.queryset = self.queryset.filter(status__ne='Rejected')

        self.queryset = self.queryset.filter(status__ne='Closed')

        self.queryset = self.queryset.filter(status__ne='Cancelled')

        self.queryset = self.queryset.order_by('-created')
        self.queryset = QuerySetHelper.Sort(self)
        return super(ChangeRequestFormsViewset, self).list(request)


    def destroy(self, request, *args, **kwargs):
        # instance = self.get_object()
        form_code = self.kwargs['form_code']
        instance = models.ChangeRequestFormHeader.objects.filter(
            form_code=form_code
        ).first()
        if instance.status.lower() == 'draft':
            models.ChangeRequestFormApprovers.objects(
                form_code=form_code
            ).delete()
            models.ChangeRequestFormDetails.objects(
                form_code=form_code
            ).delete()
            models.ChangeRequestFormStakeHolders.objects(
                form_code=form_code
            ).delete()
            models.ChangeRequestFormAttachments.objects(
                form_code=form_code
            ).delete()
            models.ChangeRequestFormTasks.objects(
                form_code=form_code
            ).delete()
            models.ChangeRequestFormHeader.objects(form_code=form_code).delete()
            # self.perform_destroy(instance)
            return Response({"message": "Change Request Deleted"}, status=status.HTTP_200_OK)
        else:
            message = {"message": "Cant delete this record, it has an ongoing tansaction"}
            return Response(
                message,
                status=status.HTTP_204_NO_CONTENT
            )

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(
            deleted_at=None,
            form_code=str(self.kwargs.get('form_code'))
        ).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)

    @action(detail=True,
        methods=['get'],
        url_path='history',
        name="CR History")
    def list_cr_history(self, request, form_code=None, **kwargs):
        
        form_code = self.kwargs['form_code']
        form_code_list = []
        form_code_list.append(form_code)

        for frmitem in models.ChangeRequestFormHeader.objects.filter(old_form_code__ne=None).order_by('created'):
            
            if frmitem.form_code == form_code:
                form_code_list.append(frmitem.old_form_code)
                form_code = frmitem.old_form_code

        # remove duplicates
        mylist = list(dict.fromkeys(form_code_list))
        print(mylist)

        queryset = models.ChangeRequestFormApprovers.objects.filter(
            (Q(action__ne=None) & Q(action__ne='')),
            form_code__in=mylist).order_by('created')

        self.serializer_class = serializers.ChangeRequestFormApproversSerializer

        page = self.paginate_queryset(queryset)
        if page is not None:
            serializer = self.get_serializer(page, many=True)

            return self.get_paginated_response(serializer.data)


class ChangeRequestFormApproversViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestFormApprovers.objects.all()
    serializer_class = serializers.ChangeRequestFormApproversSerializer
    pagination_class = paginators.SimplePageNumberPagination


    def create(self, request, *args, **kwargs):
        ObjectId = request.data.get('id')
        sent = False
        if 'form_status' in request.data:
            
            # for email
            if str(request.data['level']) == '1' and request.data['form_status'].lower() == 'pending':
                user = request.data['user']
                form_code = request.data['form_code']
                delegation = request.data['delegation']
                # initial_email(user, form_code, delegation)

                # EMAIL CODE FOR APPROVER
                notification_msg = APPROVER_MESSAGE.split(';')[0]
                email_code = APPROVER_MESSAGE.split(';')[1]

                next_appover_email(
                    user, form_code, delegation,
                    notification_msg, 'initial', email_code
                )

                request.data['date_sent'] = datetime.now().strftime('%Y-%m-%d, %H:%M:%S')
                request.data['created'] = datetime.now().strftime('%Y-%m-%d, %H:%M:%S') #correct
                sent = True

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
    
        if ObjectId:
            dbExisting = models.ChangeRequestFormApprovers.objects.filter(
                id=ObjectId,
                deleted_at=None).first()

            if dbExisting:
                dbExisting.deleted_at = datetime.now()
                dbExisting.save()

        # x = self.perform_create(serializer)
        serializer.id = None

        x = serializer.save()
        if sent == True:
            x.date_sent = datetime.now().strftime('%Y-%m-%d, %H:%M:%S')
            x.created = datetime.now().strftime('%Y-%m-%d, %H:%M:%S')  #correct
            x.save()

        headers = self.get_success_headers(serializer.data)
        return Response(
            serializer.data,
            status=status.HTTP_201_CREATED,
            headers=headers
        )

    @action(
        methods=['PATCH'], detail=False,
        url_path='approved', url_name='approved'
    )
    def approved(self, request, *args, **kwargs):
        objectid = request.data['id']
        
        instance = models.ChangeRequestFormApprovers.objects.filter(
            id=objectid
        )
        # .update(
        #     set__deleted_at=datetime.utcnow()
        # )

        instance = instance.first()
        current_remarks = instance.remarks

        print(current_remarks)

        instance.update(
            set__deleted_at=datetime.utcnow()
        )

        # request.data['date_sent'] = datetime.now().strftime('%Y-%m-%d, %H:%M:%S')
        request.data['created'] = datetime.now().strftime('%Y-%m-%d, %H:%M:%S') # correct
        # request.data['remarks'] = current_remarks

        print("hello")
        print(request.data['remarks'])
        
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)

        #------------

        id = serializer.data['id']

        current_user = request.data['user']
        form_code = request.data['form_code']
        delegation = request.data['delegation']
        action = request.data['action']
        level = request.data['level']
        next_level = int(request.data['level']) + 1
        remarks = request.data['remarks']

        next_approver = models.ChangeRequestFormApprovers.objects.filter(
                                                    level=str(next_level),
                                                    form_code=form_code,
                                                    deleted_at=None
                                                    )

        if action.lower() == 'approved':

            models.ChangeRequestFormApprovers.objects.filter(
                Q(deleted_at=None) & Q(level=str(next_level))
                ).update(
                    date_sent=datetime.now().strftime('%Y-%m-%d, %H:%M:%S')
                )

            # EMAIL CODE FOR REQUESTOR
            requestor_notification_msg = REQUESTOR_MESSAGE.split(';')[0]
            requestor_email_code = REQUESTOR_MESSAGE.split(';')[1]

            # EMAIL CODE FOR APPROVER
            notification_msg = APPROVER_MESSAGE.split(';')[0]
            email_code = APPROVER_MESSAGE.split(';')[1]
            
            send_mail_requestor(
                current_user, form_code, delegation,
                requestor_notification_msg, action, requestor_email_code,
                remarks, level
            )

            for n_approver in next_approver:
                print(n_approver.user)

                if n_approver.delegation.lower() == 'vendor/implementor':
                    notification_msg = VENDOR_ACKNOWLEDGE_MESSAGE.split(';')[0]
                    email_code = VENDOR_ACKNOWLEDGE_MESSAGE.split(';')[1]

                next_appover_email(
                    n_approver.user, form_code, delegation,
                    notification_msg, action, email_code
                )

        elif action.lower() == 'rejected':

            if delegation.lower() == 'requestor':
                notification_msg = VENDOR_REJECT_MESSAGE.split(';')[0]
                email_code = VENDOR_REJECT_MESSAGE.split(';')[1]
                
                send_mail_vendor(
                    current_user, form_code, delegation,
                    requestor_notification_msg, action, requestor_email_code,
                    remarks, level
                )
            
            else:
                # EMAIL CODE FOR REQUESTOR
                requestor_notification_msg = REQUESTOR_REJECT_MESSAGE.split(';')[0]
                requestor_email_code = REQUESTOR_REJECT_MESSAGE.split(';')[1]

                send_mail_requestor(
                    current_user, form_code, delegation,
                    requestor_notification_msg, action, requestor_email_code,
                    remarks, level
                )

        elif action.lower() == 'completed':

            models.ChangeRequestFormApprovers.objects.filter(
                Q(deleted_at=None) & Q(level=str(next_level))
                ).update(
                    date_sent=datetime.now().strftime('%Y-%m-%d, %H:%M:%S')
            )

            # EMAIL CODE FOR REQUESTOR
            requestor_notification_msg = REQUESTOR_COMPLETION_MESSAGE.split(';')[0]
            requestor_email_code = REQUESTOR_COMPLETION_MESSAGE.split(';')[1]

            send_mail_requestor(
                current_user, form_code, delegation,
                requestor_notification_msg, action, requestor_email_code,
                remarks, level
            )

        elif action.lower() == 'acknowledged':

            # models.ChangeRequestFormApprovers.objects.filter(
            #     level=str(next_level)
            #     ).update(
            #         date_sent=datetime.now().strftime('%Y-%m-%d, %H:%M:%S')
            # )

            # EMAIL CODE FOR REQUESTOR
            requestor_notification_msg = REQUESTOR_ACKNOWLEDGE_MESSAGE.split(';')[0]
            requestor_email_code = REQUESTOR_ACKNOWLEDGE_MESSAGE.split(';')[1]
            
            send_mail_requestor(
                current_user, form_code, delegation,
                requestor_notification_msg, action, requestor_email_code,
                remarks, level
            )

        elif action.lower() == 'accepted':

            # EMAIL CODE FOR VENDOR
            requestor_notification_msg = VENDOR_ACCEPTANCE_MESSAGE.split(';')[0]
            requestor_email_code = VENDOR_ACCEPTANCE_MESSAGE.split(';')[1]

            send_mail_vendor(
                current_user, form_code, delegation,
                requestor_notification_msg, action, requestor_email_code,
                remarks, level
            )

        # elif action.lower() == 'cancelled':

            # action_cancelled(self, request, *args, **kwargs)

        headers = self.get_success_headers(serializer.data)

        return Response(
            serializer.data, status=status.HTTP_201_CREATED
        )
        
        #------------

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        self.queryset = QuerySetHelper.Filter(self)
        return super(ChangeRequestFormApproversViewset, self).list(request)

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestFormAttachmentsViewset(meviewsets.ModelViewSet):
    queryset = models.ChangeRequestFormAttachments.objects.all()
    serializer_class = serializers.ChangeRequestFormAttachmentsSerializer
    pagination_class = paginators.SimplePageNumberPagination

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        self.queryset = QuerySetHelper.Filter(self)
        return super(
            ChangeRequestFormAttachmentsViewset, self
        ).list(request)

    def create(self, request, *args, **kwargs):
        attachment_no = request.data.get('attachment_no')
        ObjectId = request.data.get('id')
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        if ObjectId:
            dbExisting = models.ChangeRequestFormAttachments.objects.filter(
                id=ObjectId,
                deleted_at=None).first()

            if dbExisting:
                dbExisting.deleted_at = datetime.now()
                dbExisting.save()

        self.perform_create(serializer)

        if not attachment_no:
            id = serializer.data['id']
            db_counter = models.ChangeRequestFormAttachments.objects.all().count()
            db_counter = db_counter + 1
            models.ChangeRequestFormAttachments.objects.filter(id=id).update(
                set__attachment_no=number_generator(
                    "ATCH", db_counter
                )
            )

        headers = self.get_success_headers(serializer.data)
        return Response(
            serializer.data,
            status=status.HTTP_201_CREATED,
            headers=headers
        )

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestFormStakeHoldersViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestFormStakeHolders.objects.all()
    serializer_class = serializers.ChangeRequestFormStakeHoldersSerializer
    pagination_class = paginators.SimplePageNumberPagination

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        self.queryset = QuerySetHelper.Filter(self)
        return super(ChangeRequestFormStakeHoldersViewset, self).list(request)

    def create(self, request, *args, **kwargs):
        ObjectId = request.data.get('id')
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        if ObjectId:
            dbExisting = models.ChangeRequestFormStakeHolders.objects.filter(
                id=ObjectId,
                deleted_at=None).first()

            if dbExisting:
                dbExisting.deleted_at = datetime.now()
                dbExisting.save()

        self.perform_create(serializer)

        headers = self.get_success_headers(serializer.data)
        return Response(
            serializer.data,
            status=status.HTTP_201_CREATED,
            headers=headers
        )

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestFormDetailsViewset(meviewsets.ModelViewSet):

    queryset = models.ChangeRequestFormDetails.objects.all()
    serializer_class = serializers.ChangeRequestFormDetailsSerializer
    pagination_class = paginators.SimplePageNumberPagination

    def list(self, request, *args, **kwargs):
        self.queryset = self.queryset.filter(deleted_at=None)
        self.queryset = QuerySetHelper.Sort(self)
        self.queryset = QuerySetHelper.Filter(self)
        return super(ChangeRequestFormDetailsViewset, self).list(request)

    def create(self, request, *args, **kwargs):
        ObjectId = request.data.get('id')
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        if ObjectId:
            dbExisting = models.ChangeRequestFormDetails.objects.filter(
                id=ObjectId,
                deleted_at=None).first()

            if dbExisting:
                dbExisting.deleted_at = datetime.now()
                dbExisting.save()

        self.perform_create(serializer)

        headers = self.get_success_headers(serializer.data)
        return Response(
            serializer.data,
            status=status.HTTP_201_CREATED,
            headers=headers
        )

    def retrieve(self, request, *args, **kwargs):
        instance = self.queryset.filter(deleted_at=None).first()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


class ChangeRequestFormPost(APIView):

    # @transaction.atomic()
    def post(self, request):
        form_header = request.data

        try:
            data_list_approver = []
            data_list_stake = []
            data_list_attach = []
            data_list_detail = []

            form_header_data = {
                'requested_to_template_name': form_header['requested_to_template_name'],
                'requested_to_template_id': form_header['requested_to_template_id'],
                'requested_to_objective': form_header['requested_to_objective'],
                'requested_to_target_date': form_header['requested_to_target_date'],
                'requested_to_priority': form_header['requested_to_priority'],
                'description': form_header['description'],
                'status': form_header['status'],
                'company_desc': form_header['company_desc'],
                'department_desc': form_header['department_desc'],
                'requested_desc': form_header['requested_desc'],
                'old_form_code': form_header['old_form_code'],
                'requested_by_department': form_header['requested_by_department'],
                'requested_by_user': form_header['requested_by_user'],
                'requested_to_company': form_header['requested_to_company'],
                'requested_to_department': form_header['requested_to_department'],
                'requested_to_user': form_header['requested_to_user'],
                'template_no': form_header['template_no']
            }

            frm_approvers = form_header['frm_approvers']
            frm_stakes = form_header['frm_stakes']
            frm_attachments = form_header['frm_attachments']
            frm_details = form_header['frm_details']

            # sp1 = transaction.savepoint()  # nothing will save to db

            serializer = serializers.ChangeRequestFormHeaderSerializer(
                data=form_header_data)
            
            if serializer.is_valid(raise_exception=True):
                serializer.save()
            
            frm_id = serializer.data['form_code']
            print(frm_id)

            # create template approvers

            for frm_approver in frm_approvers:
                frm_approver['form_code'] = frm_id
                data_list_approver.append(frm_approver)

            serializerApprover = serializers.ChangeRequestFormApproversSerializer(
                data=data_list_approver, many=True)

            if serializerApprover.is_valid(raise_exception=True):
                serializerApprover.save()
            
            # create template stakes

            for frm_stake in frm_stakes:
                frm_stake['form_code'] = frm_id
                data_list_stake.append(frm_stake)

            serializerStake = serializers.ChangeRequestFormStakeHoldersSerializer(
                data=data_list_stake, many=True)

            if serializerStake.is_valid(raise_exception=True):
                serializerStake.save()
            
            # create template attachments

            for frm_attachment in frm_attachments:
                frm_attachment['form_code'] = frm_id
                data_list_attach.append(frm_attachment)

            serializerAttach = serializers.ChangeRequestFormAttachmentsSerializer(
                data=data_list_attach, many=True)

            if serializerAttach.is_valid(raise_exception=True):
                serializerAttach.save()

            # create template details
            
            for frm_detail in frm_details:
                frm_detail['form_code'] = frm_id
                data_list_detail.append(frm_detail)

            serializerDetail = serializers.ChangeRequestFormDetailsSerializer(
                data=data_list_detail, many=True)
            
            if serializerDetail.is_valid(raise_exception=True):
                serializerDetail.save()
            
            message = {
                'code': 201,
                'status': 'success',
                'message': 'Form Details successfully saved!',
                'results': serializer.data
            }

            return Response(message, status=status.HTTP_201_CREATED)

        except ValidationError as e:
            # transaction.savepoint_rollback(sp1)
            message = {
                'code': 400,
                'status': 'failed',
                'message': str(e),
            }
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            # transaction.savepoint_rollback(sp1)
            message = {
                'code': 500,
                'status': 'failed',
                'message': 'Request was not able to process' + str(e),
            }
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)