from rest_framework import status, viewsets
from rest_framework.response import Response
from app.entities import enums
from app.entities.models import ChangeRequestSettings
from app.applicationlayer.utils import CustomPagination,status_message_response
from django_filters import rest_framework as filters
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.cms.cms_settings.serializer import (
    ChangeRequestSettingsSerializer
)
from app.applicationlayer.cms.cms_settings.table_filter import (
    ChangeRequestSettingsFilterSet
)
from django.db import transaction
from rest_framework.exceptions import ValidationError
from django.db import IntegrityError


class CMSSettingsViewSet(viewsets.ModelViewSet):
    queryset = ChangeRequestSettings.objects.all()
    serializer_class = ChangeRequestSettingsSerializer
    pagination_class = CustomPagination
    lookup_field = 'code'
    filter_backends = (DjangoFilterBackend, SearchFilter, OrderingFilter)
    filterset_class = ChangeRequestSettingsFilterSet
    ordering_fields = '__all__'
    search_fields = (
        'active_user__code',
        'behalf_user__code'
    )

    def list(self, request, *args, **kwargs):

        logged_user = self.request.user.code
        queryset = self.filter_queryset(
            self.get_queryset().filter(active_user=logged_user)
        )
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of CMS Settings found',
                serializer.data
            )
            
            return self.get_paginated_response(message)

        serializer = self.get_serializer(queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )

    @transaction.atomic
    def create(self, request, *args, **kwargs):
        try:
            logged_user = self.request.user.code
            ChangeRequestSettings.objects.filter(active_user=logged_user).delete()
            data_cms_settings = []
            
            cms_settings = request.data['cms_settings']
            for cms_setting in cms_settings:
                cms_setting['active_user'] = logged_user
                data_cms_settings.append(cms_setting)

            serializer = ChangeRequestSettingsSerializer(
                data=data_cms_settings, many=True
            )
            
            if serializer.is_valid(raise_exception=True):
                serializer.save()
                
                message = {
                    'code': 201,
                    'status': 'success',
                    'message': 'CMS Settings successfully saved!',
                    'results': serializer.data
                }

                return Response(message, status=status.HTTP_201_CREATED)

        except ValidationError as e:
            message = {
                'code': 400,
                'status': 'failed',
                'message': str(e),
            }
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            message = {
                'code': 500,
                'status': 'failed',
                'message': str(e),
            }
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)
