from rest_framework import status, viewsets
from rest_framework.response import Response
from app.entities.models import AMSPrivilege
from app.applicationlayer.utils import (CustomPagination,
                                        status_message_response)
from django_filters import rest_framework as filters
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.ams.ams_privilege.serializer import (
    AMSPrivilegeSerializer
)
from app.applicationlayer.ams.ams_privilege.table_filter import (
    AMSPrivilegeFilterSet
)
from django.db import transaction
from rest_framework.exceptions import ValidationError
from django.db import IntegrityError


class AMSPrivilegeViewSet(viewsets.ModelViewSet):
    queryset = AMSPrivilege.objects.select_related(
        'id_number', 'department'
    ).all()
    serializer_class = AMSPrivilegeSerializer
    pagination_class = CustomPagination
    lookup_field = 'id_number'
    filter_backends = (DjangoFilterBackend, SearchFilter, OrderingFilter)
    filterset_class = AMSPrivilegeFilterSet
    ordering_fields = '__all__'
    search_fields = (
        'id_number__code',
        'department__code',
        'ams_user_type'
    )
    
    def list(self, request, *args, **kwargs):
        
        queryset = self.filter_queryset(self.get_queryset())
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)
            
            message = status_message_response(
                200,
                'success',
                'list of AMS Privilege found',
                serializer.data
            )
            
            return self.get_paginated_response(message)

        serializer = self.get_serializer(queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )

    @transaction.atomic
    def create(self, request, *args, **kwargs):
        try:
            body_data = request.data['body_data']
            id_number = body_data[0]['id_number']
            AMSPrivilege.objects.filter(id_number=id_number).delete()
            serializer = AMSPrivilegeSerializer(
                data=body_data, many=True
            )
            if serializer.is_valid(raise_exception=True):
                serializer.save()
                message = {
                    'code': 201,
                    'status': 'success',
                    'message': 'AMS User Privilege successfully saved!',
                    'results': serializer.data
                }

                return Response(message, status=status.HTTP_201_CREATED)

        except ValidationError as e:
            message = {
                'code': 400,
                'status': 'failed',
                'message': str(e),
            }
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except IntegrityError as e:
            error = 'Adding the same Department under the same Company is not allowed'
            message = {
                'code': 400,
                'status': 'failed',
                'message': error,
            }
            return Response(message,
                            status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            message = {
                'code': 500,
                'status': 'failed',
                'message': str(e),
            }
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)
