from app.entities.models import AMSAsset, AMSAssetType
from rest_framework import viewsets, status
from rest_framework.response import Response
from django_filters import rest_framework as filters
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.ams.asset.table_filters import AMSAssetFilter
from app.applicationlayer.ams.asset_stock.serializers import AMSAssetStockSerializer
from app.applicationlayer.ams.asset import serializers
from django.db import transaction
from app.applicationlayer.utils import(
    CustomPagination, status_message_response
)
from rest_framework.decorators import action
from app.applicationlayer.utils import log_save, enums
from app.applicationlayer.utils import model_to_dict
from django.db import IntegrityError
from django.db.models import Q
from app.helper.decorators import AssetValidation


class AMSAssetViewSet(viewsets.ModelViewSet):
    queryset = AMSAsset.objects.all()
    serializer_class = serializers.AMSAssetSerializer
    pagination_class = CustomPagination
    lookup_field = 'code'
    filter_backends = (
        DjangoFilterBackend, SearchFilter,
        OrderingFilter
    )
    ordering_fields = '__all__'
    search_filter = (
        'code',
        'name',
        'asset_type__name'
    )

    def list(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset()).filter(is_active=True)

        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Asset found!',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )


    @AssetValidation
    @transaction.atomic
    def create(self, request, *args, **kwargs):
        form = request.data
        form['created_by'] = request.user.code
        
        serializer = self.get_serializer(data=form)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)

        message = status_message_response(
            201, 'success',
            'New Asset created', serializer.data
        )

        return Response(
            message
        )

    @AssetValidation
    @transaction.atomic
    def update(self, request, *args, **kwargs):
        try:
            
            partial = kwargs.pop('partial', False)
            instance = self.get_object()
            form = request.data
            form['created_by'] = request.user.code
            serializer = self.get_serializer(instance, data=form, partial=partial)
            serializer.is_valid(raise_exception=True)

            old_instance = model_to_dict(instance)
            self.perform_update(serializer)
            new_instance = serializer.data

            log_save(
                enums.LogEnum.UPDATE.value,
                enums.LogEntitiesEnum.AMSAsset.value,
                old_instance['id'],
                old_instance,
                new_instance
            )

            return Response(serializer.data)
        except IntegrityError as e:
            return Response(
                {"message": "Cannot delete or update this reocrd it has foreign key constraint to other tables"},
                status=status.HTTP_400_BAD_REQUEST
            )

    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)

    @transaction.atomic
    def destroy(self, request, *args, **kwargs):

        instance = self.get_object()
        if instance.ams_asset_to_assetdetail.count() > 0:
            message = status_message_response(
                400, 'Failed',
                'This Asset had an Asset Stocks', model_to_dict(instance)
            )
            return Response(
                message,
                status=status.HTTP_400_BAD_REQUEST
            )
        instance.is_active = False
        instance.save()
        new_instance = model_to_dict(instance)

        log_save(
            enums.LogEnum.DELETED.value,
            enums.LogEntitiesEnum.AMSAsset.value,
            new_instance['id'],
            new_instance,
            ''
        )

        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(
        methods=['PATCH'], detail=True,
        url_path='restore', url_name='restore'
    )
    def restore(self, request, code=None):

        try:
            instance = self.get_object()
            instance.is_active = True
            instance.save()
            new_instance = model_to_dict(instance)

            log_save(
                enums.LogEnum.UPDATE.value,
                enums.LogEntitiesEnum.AMSAsset.value,
                new_instance['id'],
                new_instance,
                ''
            )

            message = status_message_response(
                200, 'success',
                'Archived Asset restored',
                ''
            )

            return Response(message, status=status.HTTP_200_OK)
        
        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @action(methods=['GET'], detail=True,
            url_path='stock-list', url_name='stock_list')
    def stock_list(self, request, code):
        self.serializer_class = AMSAssetStockSerializer
        queryset = self.get_object().ams_asset_to_assetdetail.all()
        page = self.paginate_queryset(queryset)
        
        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Archived Assets found',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )

    @action(
        methods=['GET'], detail=False,
        url_path='archived', url_name='archived'
    )
    def archived(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset())
        queryset = queryset.filter(is_active=False)

        page = self.paginate_queryset(queryset)
        
        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Archived Assets found',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )