from app.entities import models
from rest_framework import viewsets, status
from rest_framework.response import Response
from django_filters import rest_framework as filters
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.ams.report_type.table_filters import AMSReportTypeFilter
from app.applicationlayer.ams.report_type import serializers
from app.applicationlayer.utils import (
    CustomPagination, status_message_response
)
from app.helper import decorators
from django.db import transaction
from app.applicationlayer.utils import model_to_dict
from app.applicationlayer.utils import log_save, enums
from django.db import IntegrityError
from rest_framework.decorators import action


class AMSReportTypeViewset(viewsets.ModelViewSet):

    queryset = models.AMSReportType.objects.all()
    serializer_class = serializers.AMSReportTypeSerializer
    pagination_class = CustomPagination
    lookup_field = "code"
    filter_backends = (DjangoFilterBackend, SearchFilter, OrderingFilter)
    ordering_fields = '__all__'
    search_fields = (
        'code', 'name'
    )

    @transaction.atomic
    def create(self, request, *args, **kwargs):

        form = request.data
        form['created_by'] = request.user.code

        serializer = self.get_serializer(data=form)
        # serializer.is_valid(raise_exception=True)
        if serializer.is_valid(raise_exception=True):
            serializer.save(created_by=self.request.user)
            self.perform_create(serializer)

            message = status_message_response(
                201, 'success',
                'New Report Type created', serializer.data
            )

            return Response(
                message
            )


    def update(self, request, *args, **kwargs):
        try:
            
            partial = kwargs.pop('partial', False)
            instance = self.get_object()
            form = request.data
            form['created_by'] = request.user.code
            serializer = self.get_serializer(instance, data=form, partial=partial)
            serializer.is_valid(raise_exception=True)

            old_instance = model_to_dict(instance)
            self.perform_update(serializer)
            new_instance = serializer.data

            log_save(
                enums.LogEnum.UPDATE.value,
                enums.LogEntitiesEnum.AMSReportType.value,
                old_instance['id'],
                old_instance,
                new_instance
            )

            return Response(serializer.data)
        except IntegrityError as e:
            print(e)
            return Response(
                {"message": "Cannot delete or update this reocrd it has foreign key constraint to other tables"},
                status=status.HTTP_400_BAD_REQUEST
            )


    def list(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset()).filter(is_active=True)
 
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Report Types found!',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )
    
    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)

    @transaction.atomic
    def destroy(self, request, *args, **kwargs):

        instance = self.get_object()
        instance.is_active = False
        instance.save()
        new_instance = model_to_dict(instance)

        log_save(
            enums.LogEnum.UPDATE.value,
            enums.LogEntitiesEnum.AMSReportType.value,
            new_instance['id'],
            new_instance,
            ''
        )

        return Response(status=status.HTTP_204_NO_CONTENT)


    @action(
        methods=['GET'], detail=False,
        url_path='archived', url_name='archived'
    )
    def archived(self, request, *args, **kwargs):
        self.serializer_class = serializers.AMSReportTypeSerializer
        queryset = self.filter_queryset(self.queryset).filter(is_active=False)
        
        
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Archived Report type found',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )


    @action(
        methods=['PATCH'], detail=True,
        url_path='restore', url_name='restore'
    )
    def restore(self, request, code=None):
        instance = self.get_object()
        instance.is_active = True
        instance.save()
        new_instance = model_to_dict(instance)

        log_save(
            enums.LogEnum.UPDATE.value,
            enums.LogEntitiesEnum.AMSReportType.value,
            new_instance['id'],
            new_instance,
            ''
        )

        return Response(status=status.HTTP_204_NO_CONTENT)
