from app.entities import models
from rest_framework import viewsets, status
from rest_framework.response import Response
from django_filters import rest_framework as filters
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.ams.asset_type.table_filters import AssetTypeFilter
from app.applicationlayer.ams.asset_type import serializers
from app.applicationlayer.utils import (
    CustomPagination, status_message_response
)
from app.helper import decorators
from django.db import transaction
from rest_framework.decorators import action


class AssetTypeViewset(viewsets.ModelViewSet):

    queryset = models.AMSAssetType.objects.all()
    serializer_class = serializers.AssetTypeSerializer
    pagination_class = CustomPagination
    lookup_field = "code"
    filter_backends = (DjangoFilterBackend, SearchFilter, OrderingFilter)
    ordering_fields = '__all__'
    search_fields = (
        'code', 'name'
    )

    @transaction.atomic
    def create(self, request, *args, **kwargs):

        serializer = self.get_serializer(data=request.data)
        # serializer.is_valid(raise_exception=True)
        if serializer.is_valid(raise_exception=True):
            serializer.save(created_by=self.request.user)
            self.perform_create(serializer)

            message = status_message_response(
                201, 'success',
                'New Asset Type created', serializer.data
            )

            return Response(
                message
            )

    def list(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset())
        # queryset = models.AMSAssetType.objects.annotate(
        #             view_count=Sum(
        #                 When(relations_item__has_viewed=True, then=1),
        #                 output_field=IntegerField(),
        #             ),
        #             love_count=Sum(
        #                 When(relations_item__has_loved=True, then=1),
        #                 output_field=IntegerField(),
        #             ),
        #         )
 
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Asset Types found!',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )
    
    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)

    @transaction.atomic
    def destroy(self, request, *args, **kwargs):

        instance = self.get_object()
        self.perform_destroy(instance)

        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(
        methods=['GET'], detail=False,
        url_path='archived', url_name='archived'
    )
    def archived(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset())
        queryset = queryset.filter(is_active=False)
            
        page = self.paginate_queryset(queryset)
        
        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Archived Asset Types found',
                serializer.data
            )

            return self.get_paginated_response(message)
        
        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )