from rest_framework import serializers
import ast
from django.contrib.auth.hashers import make_password, check_password
import re
from django.contrib.auth import authenticate
from app.applicationlayer.utils import model_to_dict
from app.entities import enums
from django.db.models import Q
from app.applicationlayer.management.module.serializer import ModuleSerializer
from app.entities.models import (
    User, Department, UserImage, AllowedCompany, Application, Module,
    ChangeRequestSettings
)


class UserManagementRetreiveSerializer(serializers.ModelSerializer):
    image = serializers.SerializerMethodField()
    applications = serializers.SerializerMethodField()
    department = serializers.SerializerMethodField()
    company = serializers.SerializerMethodField()
    create_form = serializers.SerializerMethodField()
    create_template = serializers.SerializerMethodField()
    cms_settings = serializers.SerializerMethodField()

    class Meta:
        model = User
        fields = (
            'id',
            'create_form',
            'create_template',
            'code', 'name',
            'username', 'contact_no',
            'image', 'company',
            'department', 'applications',
            'email', 'default_app',
            'user_type', 'is_active',
            'doa',
            'cms_settings',
        )
        # exclude = ['password', 'application', 'groups', 'user_permissions']


    def get_cms_settings(self, user):

        data = user.cms_settings_behalf_users.filter(
            behalf_user=user.code
        ).values(
            'active_user__name', 'active_user__code',
            "behalf_approved", 'behalf_accepted',
            'behalf_rejected', 'behalf_implemented',
            'behalf_completed'
        )

        return data



    def get_department(self, user):
        data = {
            "id": user.department.id,
            "code": user.department.code,
            "name": user.department.name
        }
        return data

    def get_company(self, user):
        data = {
            "id": user.department.company.id,
            "code": user.department.company.code,
            "name": user.department.company.name
        }
        return data

    def get_create_form(self, user):
        check_user = AllowedCompany.objects.filter(
            Q(id_number=user.code) &
            Q(create_change_request=True)
        ).count()
        if check_user <= 0:
            permission = False
        else:
            permission = True
        return permission

    def get_create_template(self, user):
        check_user = AllowedCompany.objects.filter(
            Q(id_number=user.code) &
            Q(create_change_request_template=True)
        ).count()
        if check_user <= 0:
            permission = False
        else:
            permission = True
        return permission

    def get_image(self, user):
        request = self.context.get('request')

        primayPic = user.user_images.filter(is_primary=True).first()
        if primayPic:
            path = request.build_absolute_uri(primayPic.image.url)
            path = path.replace('http', 'https')
            return path
        else:
            return request.build_absolute_uri('/media/no-user.png')

    def get_applications(self, user):
        if user.user_type.upper() == enums.UserTypeEnum.USER.value:
            app = user.application.exclude(id=1)
        else:
            rms = Application.objects.filter(id=1)
            app = user.application.all()
            app = app.union(app, rms)
        list_app = []

        for data in app.order_by('id'):
            if data.id == 1 and user.user_type.upper() == enums.UserTypeEnum.SUPER_USER.value:
                mod = data.modules.all()
            elif data.id == 1 and user.user_type.upper() != enums.UserTypeEnum.USER.value:
                mod = Module.objects.filter(
                        name__icontains="user"
                    )
            else:
                mod = data.modules.all()
            
            if user.user_type.upper() != enums.UserTypeEnum.USER.value:
                user_module = Module.objects.filter(name__icontains="user")
                mod = mod.union(mod, user_module)
            else:
                mod = mod
            mod = mod.order_by("parent", "sort_id")

            mod = ModuleSerializer(data=mod, many=True)
            mod.is_valid()

            app_body = {}
            app_body['app_code'] = data.code
            app_body['name'] = data.name
            app_body['modules'] = mod.data
            list_app.append(app_body)
        return list_app


class UserSerializer(serializers.ModelSerializer):

    # department = serializers.CharField(write_only=True)

    def to_representation(self, instance):
        ret = super().to_representation(instance)
        get_code = User.objects.get(id=instance.id)
        ret['department'] = model_to_dict(instance.department)
        ret['code'] = str(get_code.code)
        ret['company'] = model_to_dict(instance.department.company)
        application = instance.application.values()
        ret['application'] = application

        return ret
    class Meta:
        model = User
        fields = (
            'id',
            'code', 'name',
            'username', 'contact_no',
            'email', 'default_app',
            'user_type', 'is_active',
            'doa', 'department'

        )

        read_only_fields = (
            'created', 'createdby', 'modified', 'modifiedby', 'code',
        )


class UserEditSerializer(serializers.ModelSerializer):

    class Meta:
        model = User
        fields = (
            'name',
            'contact_no'
        )


class ChangeRequestList(serializers.ModelSerializer):

    def to_representation(self, instance):
        ret = super().to_representation(instance)
        
        ret['company_name'] = instance.department.company.name
        ret['company_code'] = instance.department.company.code
        ret['department_code'] = instance.department.code
        ret['department_name'] = instance.department.name
        return ret

    class Meta:
        model = User
        fields = (
            'id',
            'code', 'name',
            'username', 'contact_no',
            'email', 'default_app',
            'user_type', 'is_active',
            'doa'

        )

class ChangePasswordSerializer(serializers.Serializer):
    old_password = serializers.CharField(required=True)
    new_password = serializers.CharField(required=True, min_length=6)
    new_password_confirm = serializers.CharField(required=True, min_length=6)

    def validate(self, data):

        instance = self.context.get('view').kwargs['code']
        old_password = data['old_password']
        new_password = data['new_password']

        instance_password = User.objects.filter(
            code=str(instance)
        )

        validated_password = check_password(
            old_password,
            instance_password.values().first()['password']
        )

        if validated_password:

            # password = re.match(
            #     '([A-Za-z]+[0-9]|[0-9]+[A-Za-z])[A-Za-z0-9]*',
            #     new_password
            # )

            # if password:

            #     new_password = make_password(new_password)
            #     instance_password.update(password=new_password)
            #     return instance

            if len(new_password) <= 5:

                raise serializers.ValidationError(
                    'Password must be minimum of 6 characters'
                )

            else:
                new_password = make_password(new_password)
                instance_password.update(password=new_password)
                return instance

            # else:

            #     raise serializers.ValidationError(
            #         'password must be alpha numeric format'
            #     )
        else:
            raise serializers.ValidationError('Please ensure the old password is correct and both new password & confirm password are the same.')


class UserImageSerializer(serializers.ModelSerializer):
    class Meta:
        model = UserImage
        fields = '__all__'
        read_only_fields = ['createdby', 'modifiedby', 'created', 'modified']
