from app.entities.models import AMSAssetStock
from rest_framework import viewsets, status
from rest_framework.response import Response
from django_filters import rest_framework as filters
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.ams.asset_stock.table_filters import AMSAssetStockFilter
from app.applicationlayer.ams.asset_stock import serializers
from django.db import transaction
from app.applicationlayer.utils import(
    CustomPagination, status_message_response
)
from rest_framework.decorators import action
from app.applicationlayer.utils import log_save, enums
from app.applicationlayer.utils import model_to_dict
from django.db import IntegrityError
from django.db.models import Q
from app.helper.decorators import AssetStockValidation
from django.db.models import Count


class AMSAssetStockViewSet(viewsets.ModelViewSet):
    queryset = AMSAssetStock.objects.all()
    serializer_class = serializers.AMSAssetStockSerializer
    pagination_class = CustomPagination
    lookup_field = 'code'
    filter_backends = (
        DjangoFilterBackend, SearchFilter,
        OrderingFilter
    )
    ordering_fields = '__all__'
    search_filter = (
        'code',
        'name',
        'asset_group__name',
        'asset__name',
        'asset_type__name',
    )

    def list(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset()).filter(is_active=True)

        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Asset Stock list found!',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )
    
    @action(detail=False,
            methods=['get'],
            url_path='dashboard',
            name="Dashboard Summary")
    def dashboard_asset(self, request):
        
        in_store = self.queryset.filter(
            status__iexact='In-Store'
        ).count()

        deployed = self.queryset.filter(
            status__iexact='Deployed'
        ).count()

        returned = self.queryset.filter(
            status__iexact='Returned'
        ).count()

        written_off = self.queryset.filter(
            status__iexact='Written Off'
        ).count()
        
        faulty = self.queryset.filter(
            status__iexact='Faulty'
        ).count()
        
        in_qty = self.queryset.filter(
            status__iexact='In-Store'
        ).count()

        out_qty = self.queryset.filter(
            status__iexact='Deployed'
        ).count()

        unavailable = self.queryset.filter(
            status__iexact=['Faulty', 'Returned', 'Written Off']
        ).count()

        top_asset_groups = self.queryset.values('asset_group__name', 'asset_group', 'asset').annotate(
            asset_type_count=Count('asset__asset_type'),
            asset_count=Count('asset'),
            asset_stock_count=Count('asset_group')).order_by('-asset_group')

        recent_reports = 0

        message = {
            'in_store': in_store,
            'deployed': deployed,
            'returned': returned,
            'written_off': written_off,
            'faulty': faulty,
            'in_qty': in_qty,
            'out_qty': out_qty,
            'unavailable': unavailable,
            'top_asset_groups': top_asset_groups,
            'recent_reports': recent_reports,
            'code': 200,
            'status': 'success',
            'message': 'Dashboard Summary'
        }

        return Response(message, status=status.HTTP_200_OK)

    @AssetStockValidation
    @transaction.atomic
    def update(self, request, *args, **kwargs):
        try:
            partial = kwargs.pop('partial', False)
            instance = self.get_object()
            form = request.data
            form['created_by'] = request.user.code
            serializer = self.get_serializer(instance, data=form, partial=partial)
            serializer.is_valid(raise_exception=True)

            old_instance = model_to_dict(instance)
            self.perform_update(serializer)
            new_instance = serializer.data

            log_save(
                enums.LogEnum.UPDATE.value,
                enums.LogEntitiesEnum.AMSReport.value,
                old_instance['id'],
                old_instance,
                new_instance
            )

            return Response(serializer.data)
        except IntegrityError as e:
            return Response(
                {"message": "Cannot delete or update this reocrd it has foreign key constraint to other tables"},
                status=status.HTTP_400_BAD_REQUEST
            )


    @AssetStockValidation
    @transaction.atomic
    def create(self, request, *args, **kwargs):

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)

        message = status_message_response(
            201, 'success',
            'New Asset Stock created', serializer.data
        )

        return Response(
            message
        )


    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)

    @transaction.atomic
    def destroy(self, request, *args, **kwargs):

        instance = self.get_object()
        instance.is_active = False
        instance.save()
        new_instance = model_to_dict(instance)

        log_save(
            enums.LogEnum.UPDATE.value,
            enums.LogEntitiesEnum.AMSAssetStock.value,
            new_instance['id'],
            new_instance,
            ''
        )

        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(
        methods=['GET'], detail=False,
        url_path='archived', url_name='archived'
    )
    def archived(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset())
        queryset = queryset.filter(is_active=False)
            
        page = self.paginate_queryset(queryset)
        
        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'List of Archived Asset Stocks found',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(self.queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )

    @action(
        methods=['PATCH'], detail=True,
        url_path='restore', url_name='restore'
    )
    def restore(self, request, code=None):

        try:
            instance = self.get_object()
            instance.is_active = True
            instance.save()
            new_instance = model_to_dict(instance)

            log_save(
                enums.LogEnum.UPDATE.value,
                enums.LogEntitiesEnum.AMSAssetStock.value,
                new_instance['id'],
                new_instance,
                ''
            )

            message = status_message_response(
                200, 'success',
                'Archived Asset Stock restored',
                ''
            )

            return Response(message, status=status.HTTP_200_OK)
        
        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)
