from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from api.models import APIEndpoint, APIService
from api.serializers import APIEndpointSerializer
from api.utils import (
            CustomPagination, BadRequestException,
            status_message_response, number_generator, tbl_ordering
        )
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter


class APIEndpointViewSet(viewsets.ModelViewSet):
    http_method_names = [
        'get', 'post', 'put', 'patch', 'delete', 'head', 'options', 'trace'
    ]
    queryset = APIEndpoint.objects.all().order_by('service')
    serializer_class = APIEndpointSerializer
    lookup_field = 'pk'
    pagination_class = CustomPagination
    filter_backends = (DjangoFilterBackend, SearchFilter,)
    search_fields = (
            'api_endpoint_no', 'service__name', 'name', 'description',
            'http_method', 'endpoint_url'
        )
    # filter_class = APIEndpointFilter

    # CREATE Endpoint
    def create(self, request, *args, **kwargs):
        try:
            serializer = self.get_serializer(data=request.data)
            serializer.is_valid(raise_exception=True)
            endpoint_create = serializer.save()

            # for endpoint_no
            endpoint_id = endpoint_create.pk
            endpoint_create.api_endpoint_no = number_generator(
                                            'ENP', endpoint_id
                                        )
            endpoint_create.save()

            message = status_message_response(
                            201, 'success',
                            'New endpoint created', serializer.data
                        )
            return Response(message)

        except BadRequestException as e:
            message = status_message_response(400, 'failed', str(e), [])
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SHOW LIST of endpoints
    def list(self, request, *args, **kwargs):
        try:
            queryset = self.queryset.filter(deleted_at__exact=None)

            # table ordering
            if 'sort_order' in request.query_params and 'sort_field' in request.query_params:
                queryset = tbl_ordering(
                                    queryset, **request.query_params
                                )
            queryset = self.filter_queryset(queryset)

            if not queryset.exists():
                message = status_message_response(
                                200, 'success', 'No records found', []
                            )
                return Response(message)

            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                enp_data = serializer.data
                for service in enp_data:
                    svc_values = APIService.objects.filter(
                                        id=service['service']).values()
                    service['service'] = svc_values[0]

                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'List of endpoints found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SHOW endpoint details
    def retrieve(self, request, *args, **kwargs):
        try:
            id = self.kwargs['pk']
            instance = APIEndpoint.objects.get(id=id)
            serializer = self.get_serializer(instance)

            message = status_message_response(
                            200, 'success',
                            'Endpoint retrieved', serializer.data
                          )
            return Response(message, status=status.HTTP_200_OK)

        except APIEndpoint.DoesNotExist:
            message = status_message_response(404, 'failed',
                                              'No record found', [])
            return Response(message, status=status.HTTP_404_NOT_FOUND)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # UPDATE endpoint
    def update(self, request, *args, **kwargs):
        try:
            partial = kwargs.pop('partial', False)
            instance = self.get_object()
            serializer = self.get_serializer(
                                instance, data=request.data, partial=partial
                            )
            serializer.is_valid(raise_exception=True)
            self.perform_update(serializer)
            if getattr(instance, '_prefetched_objects_cache', None):
                instance._prefetched_objects_cache = {}

            message = status_message_response(
                            200, 'success',
                            'Endpoint updated', serializer.data
                        )
            return Response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SOFT DELETE / ARCHIVED
    def destroy(self, request, *args, **kwargs):
        try:
            instance = self.get_object()
            self.perform_destroy(instance)
            message = status_message_response(
                            200, 'success', 'Endpoint deleted', []
                        )
            return Response(message, status=status.HTTP_200_OK)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # PATCH - RESTORE archived endpoint
    def partial_update(self, request, *args, **kwargs):
        try:
            kwargs['partial'] = True
            instance = self.get_object()
            instance.deleted_at = None
            serializer = self.get_serializer(instance)
            message = status_message_response(
                            200, 'success',
                            'Archived endpoint restored', serializer.data
                        )
            instance.save()
            return Response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # /archived - show list of archived endpoints
    @action(methods=["GET"], detail=False)
    def archived(self, request, pk=None):
        try:
            queryset = APIEndpoint.objects.filter(deleted_at__isnull=False)
            # table ordering
            if 'sort_order' in request.query_params and 'sort_field' in request.query_params:
                queryset = tbl_ordering(
                                    queryset, **request.query_params
                                )
            queryset = self.filter_queryset(queryset)

            if not queryset.exists():
                message = status_message_response(
                                200, 'success', 'No archived endpoints', []
                            )
                return Response(message)

            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'Archived endpoints found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)
