import requests
from django.conf import settings
from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from api.models import Application
from api.serializers import ApplicationSerializer, GroupDependentSerializer
from api.utils import (
            CustomPagination, BadRequestException,
            status_message_response, number_generator, tbl_ordering
        )
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter


AUTHENTICATOR_GROUP = settings.AUTHENTICATOR_GROUP


class ApplicationViewSet(viewsets.ModelViewSet):
    http_method_names = [
        'get', 'post', 'put', 'patch', 'delete', 'head', 'options', 'trace'
    ]
    lookup_field = 'pk'
    queryset = Application.objects.all()
    serializer_class = ApplicationSerializer
    pagination_class = CustomPagination
    filter_backends = (DjangoFilterBackend, SearchFilter,)
    search_fields = ('application_no', 'name', 'code')
    # filter_class = APIEndpointFilter

    # CREATE Application
    def create(self, request, *args, **kwargs):
        try:
            serializer = self.get_serializer(data=request.data)
            serializer.is_valid(raise_exception=True)
            app_create = serializer.save()

        # for application_no
            app_id = app_create.pk
            app_create.application_no = number_generator(
                                            'APP', app_id
                                        )
            app_create.save()

            message = status_message_response(
                201, 'success',
                'New application created', serializer.data
            )
            return Response(message)

        except BadRequestException as e:
            message = status_message_response(400, 'failed', str(e), [])
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            message = status_message_response(
                500, 'failed',
                'Request was not able to process' + str(e), []
            )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SHOW LIST of Applications
    def list(self, request, *args, **kwargs):
        try:
            # queryset = Application.objects.filter(deleted_at__exact=None)
            queryset = self.queryset.filter(deleted_at__exact=None)
            queryset = self.filter_queryset(queryset)
            ids = self.request.query_params.get(
                'ids', None
            )

            # table ordering
            if 'sort-field' and 'sort-order' in request.query_params:
                queryset = tbl_ordering(
                                    queryset, **request.query_params
                                )

            if not queryset.exists():
                message = status_message_response(
                    200, 'success', 'No records found', []
                )
                return Response(message)
            if ids is not None:
                ids = ids.split(',')
                queryset = queryset.filter(id__in=ids)

            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                results = []
                for item in serializer.data:
                    # print(item['code'])
                    ids = item['code']
                    # headers = {
                    #     'content-type': 'application/json',
                    #     'Authorization': request.META['HTTP_AUTHORIZATION']
                    # }
                    req = requests.get(f'{AUTHENTICATOR_GROUP}/{ids}/')
                    groups = req.json()['groups']

                    modules = req.json()['modules']
                    item['groups'] = groups
                    item['modules'] = modules
                    results.append(item)

                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'List of applications found',
                    'results': results
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                500, 'failed',
                'Request was not able to process' + str(e), [])
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SHOW application details
    def retrieve(self, request, *args, **kwargs):
        try:
            id = self.kwargs['pk']
            instance = Application.objects.get(id=id)
            serializer = self.get_serializer(instance)

            message = status_message_response(
                            200, 'success',
                            'Application retrieved', serializer.data
                          )
            return Response(message, status=status.HTTP_200_OK)

        except Application.DoesNotExist:
            message = status_message_response(404, 'failed',
                                              'No record found', [])
            return Response(message, status=status.HTTP_404_NOT_FOUND)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # UPDATE Application
    def update(self, request, *args, **kwargs):
        try:
            partial = kwargs.pop('partial', False)
            instance = self.get_object()
            serializer = self.get_serializer(instance, data=request.data,
                                             partial=partial)
            serializer.is_valid(raise_exception=True)
            self.perform_update(serializer)
            if getattr(instance, '_prefetched_objects_cache', None):
                instance._prefetched_objects_cache = {}

            message = status_message_response(
                            200, 'success', 'Application updated',
                            serializer.data
                        )
            return Response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SOFT DELETE / ARCHIVED
    def destroy(self, request, *args, **kwargs):
        try:
            instance = self.get_object()
            self.perform_destroy(instance)

            message = status_message_response(
                            200, 'success', 'Application deleted', []
                        )
            return Response(message, status=status.HTTP_200_OK)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # PATCH - RESTORE archived application
    def partial_update(self, request, *args, **kwargs):
        try:
            kwargs['partial'] = True
            instance = self.get_object()
            instance.deleted_at = None
            serializer = self.get_serializer(instance)
            message = status_message_response(
                            200, 'success',
                            'Archived application restored', serializer.data
                        )
            instance.save()
            return Response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # /archived - show list of archived application
    @action(methods=["GET"], detail=False)
    def archived(self, request, pk=None):
        try:
            queryset = Application.objects.filter(deleted_at__isnull=False)

            if not queryset.exists():
                message = status_message_response(
                                200, 'success',
                                'No archived applications', []
                            )
                return Response(message)

            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'Archived applications found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # Show app services
    @action(methods=['GET'], detail=True)
    def services(self, request, pk):
        try:
            services = APIService.objects.filter(application=pk)

            page = self.paginate_queryset(services)
            if page is not None:
                serializer = APIServiceSerializer(page, many=True)
                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'Application services found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class MainApplicationViewSet(viewsets.ModelViewSet):
    queryset = Application.objects.all()
    serializer_class = GroupDependentSerializer
