import requests
from rest_framework import serializers
from .models import APIService, APIEndpoint, Application
from api.utils import BadRequestException, number_generator
from rest_framework.exceptions import ValidationError
from django.utils.crypto import get_random_string
from django.conf import settings

ACCOUNT_GROUP = settings.ACCOUNT_GROUP


class ApplicationSerializer(serializers.ModelSerializer):
    groups = serializers.ListField(read_only=True)
    modules = serializers.ListField(read_only=True)

    def to_representation(self, data):
        ids = data.id
        req = requests.get(f'{ACCOUNT_GROUP}/{ids}/')
        groups = req.json()['groups']
        modules = req.json()['modules']

        setattr(data, 'groups', groups)
        setattr(data, 'modules', modules)

        return super().to_representation(data)

    class Meta:
        model = Application
        fields = ('id', 'application_no', 'name', 'theme', 'groups', 'modules')
        read_only_fields = (
            'id', 'application_no', 'created_at', 'updated_at', 'deleted_at'
        )

    def is_valid(self, raise_exception=False):

        assert not hasattr(self, 'restore_object'), (
            'Serializer `%s.%s` has old-style version 2 `.restore_object()` '
            'that is no longer compatible with REST framework 3. '
            'Use the new-style `.create()` and `.update()` methods instead.' %
            (self.__class__.__module__, self.__class__.__name__)
        )

        assert hasattr(self, 'initial_data'), (
            'Cannot call `.is_valid()` as no `data=` keyword argument was '
            'passed when instantiating the serializer instance.'
        )

        if not hasattr(self, '_validated_data'):
            try:
                self._validated_data = self.run_validation(self.initial_data)
            except ValidationError as exc:
                self._validated_data = {}
                self._errors = exc.detail
            else:
                self._errors = {}

        #if validation failed
        if self._errors and raise_exception:
            error_message = {}
            message = str(self.errors)

            for k, v in self.errors.items():
                message = str(v)
                start = message.find('string=') + 8
                end = message.find(', code=') - 1
                message = message[start:end]

                error_message[str(k)] = message

            raise BadRequestException(error_message)

        return not bool(self._errors)

    def create(self, validated_data):

        new_application = Application.objects.create(**validated_data)
        new_application.application_no = number_generator('APP', new_application.id)
        new_application.save()
        return new_application


class APIServiceSerializer(serializers.ModelSerializer):
    class Meta:
        model = APIService
        fields = ('id', 'api_service_no', 'name', 'service_token', 'base_url', 'service_url', 'application', 'created_at', 'updated_at', 'deleted_at')
        read_only_fields = ('id', 'api_service_no', 'service_token', 'created_at', 'updated_at', 'deleted_at')

    def is_valid(self, raise_exception=False):

        assert not hasattr(self, 'restore_object'), (
            'Serializer `%s.%s` has old-style version 2 `.restore_object()` '
            'that is no longer compatible with REST framework 3. '
            'Use the new-style `.create()` and `.update()` methods instead.' %
            (self.__class__.__module__, self.__class__.__name__)
        )

        assert hasattr(self, 'initial_data'), (
            'Cannot call `.is_valid()` as no `data=` keyword argument was '
            'passed when instantiating the serializer instance.'
        )

        if not hasattr(self, '_validated_data'):
            try:
                self._validated_data = self.run_validation(self.initial_data)
            except ValidationError as exc:
                self._validated_data = {}
                self._errors = exc.detail
            else:
                self._errors = {}

        #if validation failed
        if self._errors and raise_exception:
            error_message = {}
            message = str(self.errors)

            for k, v in self.errors.items():
                message = str(v)
                start = message.find('string=') + 8
                end = message.find(', code=') - 1
                message = message[start:end]

                error_message[str(k)] = message

            raise BadRequestException(error_message)

        return not bool(self._errors)

    def create(self, validated_data):
        new_service = APIService.objects.create(**validated_data)
        new_service.api_service_no = number_generator('SVC', new_service.id)
        new_service.service_token = get_random_string(length=16)
        new_service.save()
        return new_service


class APIEndpointSerializer(serializers.ModelSerializer):
    service_name = serializers.ReadOnlyField(source='service.name')

    class Meta:
        model = APIEndpoint
        fields = (
            'id', 'api_endpoint_no', 'service', 'service_name', 'name', 'description',
            'http_method', 'endpoint_url', 'is_need_auth', 'is_active', 'created_at', 'updated_at', 'deleted_at'
        )
        read_only_fields = ('id', 'api_endpoint_no', 'created_at', 'updated_at', 'deleted_at')

    def is_valid(self, raise_exception=False):

        assert not hasattr(self, 'restore_object'), (
            'Serializer `%s.%s` has old-style version 2 `.restore_object()` '
            'that is no longer compatible with REST framework 3. '
            'Use the new-style `.create()` and `.update()` methods instead.' %
            (self.__class__.__module__, self.__class__.__name__)
        )

        assert hasattr(self, 'initial_data'), (
            'Cannot call `.is_valid()` as no `data=` keyword argument was '
            'passed when instantiating the serializer instance.'
        )

        if not hasattr(self, '_validated_data'):
            try:
                self._validated_data = self.run_validation(self.initial_data)
            except ValidationError as exc:
                self._validated_data = {}
                self._errors = exc.detail
            else:
                self._errors = {}

        #if validation failed
        if self._errors and raise_exception:
            error_message = {}
            message = str(self.errors)

            for k, v in self.errors.items():
                message = str(v)
                start = message.find('string=') + 8
                end = message.find(', code=') - 1
                message = message[start:end]

                error_message[str(k)] = message

            raise BadRequestException(error_message)

        return not bool(self._errors)

    def create(self, validated_data):
        new_endpoint = APIEndpoint.objects.create(**validated_data)
        new_endpoint.api_endpoint_no = number_generator('ENP', new_endpoint.id)
        new_endpoint.save()
        return new_endpoint
