from app.entities import enums
from django.db import transaction
from app.helper.decorators import rms, error_safe
from rest_framework import viewsets, status
from rest_framework.response import Response
from django.forms.models import model_to_dict
from rest_framework.filters import SearchFilter, OrderingFilter
from django_filters import rest_framework as filters
from django.contrib.auth.hashers import make_password
from app.entities.models import User, EntityLog
from app.applicationlayer.utils import (
    CustomPagination, status_message_response, log_save
)
from django_filters.rest_framework import DjangoFilterBackend
from app.applicationlayer.management.account import serializer
from app.applicationlayer.management.account.table_filters import UserFilterSet


class UserViewSet(viewsets.ModelViewSet):
    queryset = User.objects.all()
    serializer_class = serializer.UserSerializer
    pagination_class = CustomPagination
    filter_backends = (DjangoFilterBackend, SearchFilter, OrderingFilter)
    filterset_class = UserFilterSet
    ordering_fields = '__all__'
    search_fields = ('name',)

    # @check.user_type
    @rms.user_create
    @transaction.atomic
    def create(self, request, *args, **kwargs):
        password = make_password(request.data['password'])
        request.data['password'] = password
        
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)

        message = status_message_response(
            201, 'success',
            'New Users created', serializer.data
        )

        return Response(
            message
        )

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'list of Users found',
                serializer.data
            )
            return self.get_paginated_response(message)

        serializer = self.get_serializer(queryset, many=True)
        return Response(serializer.data)


    @error_safe
    @transaction.atomic
    def destroy(self, request, *args, **kwargs):

        instance = self.get_object()
        new_instance = model_to_dict(instance)
        self.perform_destroy(instance)

        log_save(
            enums.LogEnum.DELETED.value,
            enums.LogEntitiesEnum.USER.value,
            new_instance['id'],
            new_instance,
            ''
        )

        return Response(status=status.HTTP_204_NO_CONTENT)

    @transaction.atomic
    def update(self, request, *args, **kwargs):

        partial = kwargs.pop('partial', False)
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data, partial=partial)
        serializer.is_valid(raise_exception=True)

        old_instance = model_to_dict(instance)
        self.perform_update(serializer)
        new_instance = serializer.data

        log_save(
            enums.LogEnum.UPDATE.value,
            enums.LogEntitiesEnum.USER.value,
            new_instance['id'],
            old_instance,
            new_instance
        )

        return Response(serializer.data)
