import json
import requests
from django.conf import settings
from django.http import Http404
from rest_framework import status
from django_filters import FilterSet
from django_filters import rest_framework as filters
from .models import APIEndpoint
from rest_framework.pagination import PageNumberPagination
from rest_framework.response import Response
import datetime


VALIDATE_TOKEN_URL = settings.VALIDATE_TOKEN_URL

class BadRequestException(Exception):
    pass 
    
class APIEndpointFilter(FilterSet):
    service = filters.CharFilter('service__name')

    class Meta:
        model = APIEndpoint
        fields = ('service',)


class Helper:

    def __init__(self):
        self._headers = {'Content-Type': 'application/json'}

    def _response_data(self, data, status_code, headers):
        return {
            'data': data,
            'status_code': status_code,
            'headers': headers
        }

    def _request_method(self, request, final_endpoint, headers):
        if request.method == 'GET':
            req = requests.get(final_endpoint, headers=headers)
            return self._response_data(req.json(),
                                       req.status_code,
                                       self._headers)
        elif request.method == 'POST':
            data = request.data
            json_body = json.dumps(data)
            req = requests.post(
                final_endpoint,
                data=json_body,
                headers=headers
            )
            return self._response_data(req.json(),
                                       req.status_code,
                                       self._headers)
        elif request.method == 'PUT':
            data = request.data
            json_body = json.dumps(data)
            req = requests.put(
                final_endpoint,
                data=json_body,
                headers=headers
            )
            return self._response_data(req.json(),
                                       req.status_code,
                                       self._headers)
        elif request.method == 'PATCH':
            data = request.data
            json_body = json.dumps(data)
            req = requests.patch(
                final_endpoint,
                data=json_body,
                headers=headers
            )
            return self._response_data(req.json(),
                                       req.status_code,
                                       self._headers)
        elif request.method == 'DELETE':
            req = requests.delete(final_endpoint, headers=headers)
            return self._response_data({'message': 'Deleted'},
                                       req.status_code,
                                       self._headers)

    def get_endpoint(self, request, service, endpoint_url):
        try:
            endpoint = APIEndpoint.objects.select_related('service').get(
                http_method=request.method,
                service__name=service,
                endpoint_url=endpoint_url,
            )
            if endpoint.is_active is True:
                base_url = endpoint.service.base_url
                full_path = request.get_full_path()
                final_endpoint = f"{base_url}{full_path}"
                if endpoint.is_need_auth is True:
                    # redirect to authenticator service
                    self._headers = {
                        'Authorization': request.META['HTTP_AUTHORIZATION'],
                        "endpoint": endpoint.endpoint_url
                    }
                    req = requests.post(VALIDATE_TOKEN_URL,
                                        headers=self._headers)
                    all_headers = {**req.headers, **self._headers}

                    if req.status_code == 200:
                        return self._request_method(request,
                                                    final_endpoint,
                                                    all_headers)
                    return self._response_data({'message': 'Unauthorized'},
                                               status.HTTP_401_UNAUTHORIZED,
                                               self._headers)
                else:
                    return self._request_method(request, final_endpoint,
                                                self._headers)
            else:
                return self._response_data({'message': 'Invalid'},
                                           status.HTTP_500_INTERNAL_SERVER_ERROR,
                                           self._headers)
        except APIEndpoint.DoesNotExist:
            raise Http404
        except Exception as e:
            return self._response_data({'message': str(e)},
                                       status.HTTP_500_INTERNAL_SERVER_ERROR,
                                       self._headers)

class CustomPagination(PageNumberPagination):
    page_size = 5 # Set number of result to display per page
    max_page_size = 50  # Sets max page size that user may request
    page_query_param = 'page'
    page_size_query_param = 'page_size'

    def get_paginated_response(self, data):
        return Response({
            'page_number': self.page.number,
            'size_per_page': self.page.paginator.per_page,
            'total_pages': self.page.paginator.num_pages,
            'total': self.page.paginator.count,
            'code': data['code'],
            'status': data['status'],
            'message': data['message'],
            'results': data['results']
        })

# autogenerated number
def number_generator(prefix, id):
    date = '{:%Y%m%d}'.format(datetime.datetime.now())
    id_num = '{:07}'.format(id)
    autogenerated_no = prefix + '-' + date + '-' + id_num

    return autogenerated_no

# status message
def status_message_response(code, status, message, results):
    message = {
            'code': code,
            'status': status,
            'message': message,
            'results': results
        }
    return message

