from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from api.models import APIService, Application
from api.serializers import APIServiceSerializer
from api.utils import (
            CustomPagination, BadRequestException,
            status_message_response, number_generator, tbl_ordering
        )
from django.utils.crypto import get_random_string
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter


class APIServiceViewSet(viewsets.ModelViewSet):
    http_method_names = [
        'get', 'post', 'put', 'patch', 'delete', 'head', 'options', 'trace'
    ]
    queryset = APIService.objects.all()
    serializer_class = APIServiceSerializer
    lookup_field = 'pk'
    pagination_class = CustomPagination
    filter_backends = (DjangoFilterBackend, SearchFilter,)
    search_fields = (
        'api_service_no', 'name', 'service_url', 'application__name'
    )

    # CREATE Service
    def create(self, request, *args, **kwargs):
        try:
            serializer = self.get_serializer(data=request.data)
            serializer.is_valid(raise_exception=True)
            service_create = serializer.save()

            # for service_no
            service_id = service_create.pk
            service_create.api_service_no = number_generator(
                                            'SVC', service_id
                                        )
            service_create.service_token = get_random_string(length=16)
            service_create.save()

            message = status_message_response(
                            201, 'success',
                            'New service created', serializer.data
                        )
            return Response(message)

        except BadRequestException as e:
            message = status_message_response(400, 'failed', str(e), [])
            return Response(message, status=status.HTTP_400_BAD_REQUEST)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SHOW LIST of services
    def list(self, request, *args, **kwargs):

        try:
            queryset = self.queryset.filter(deleted_at__exact=None)

            # table ordering
            if 'sort_order' in request.query_params and 'sort_field' in request.query_params:
                queryset = tbl_ordering(
                                    queryset, **request.query_params
                                )
            queryset = self.filter_queryset(queryset)

            if not queryset.exists():
                message = status_message_response(
                                200, 'success', 'No records found', []
                            )
                return Response(message)

            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                svc_data = serializer.data
                for application in svc_data:
                    app_values = Application.objects.filter(
                                        id=application['application']).values()
                    application['application'] = app_values[0]

                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'List of services found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SHOW service details
    def retrieve(self, request, *args, **kwargs):
        try:
            id = self.kwargs['pk']
            instance = APIService.objects.get(id=id)
            serializer = self.get_serializer(instance)

            message = status_message_response(
                            200, 'success',
                            'Service retrieved', serializer.data
                          )
            return Response(message, status=status.HTTP_200_OK)

        except APIService.DoesNotExist:
            message = status_message_response(404, 'failed',
                                              'No record found', [])
            return Response(message, status=status.HTTP_404_NOT_FOUND)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # UPDATE service
    def update(self, request, *args, **kwargs):
        try:
            partial = kwargs.pop('partial', False)
            instance = self.get_object()
            serializer = self.get_serializer(
                                instance, data=request.data, partial=partial
                            )
            serializer.is_valid(raise_exception=True)
            self.perform_update(serializer)
            if getattr(instance, '_prefetched_objects_cache', None):
                instance._prefetched_objects_cache = {}

            message = status_message_response(
                            200, 'success',
                            'Service updated', serializer.data
                        )
            return Response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # SOFT DELETE / ARCHIVED
    def destroy(self, request, *args, **kwargs):
        try:
            instance = self.get_object()
            self.perform_destroy(instance)

            message = status_message_response(
                            200, 'success', 'Service deleted', []
                        )
            return Response(message, status=status.HTTP_200_OK)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # PATCH - RESTORE archived service
    def partial_update(self, request, *args, **kwargs):
        try:
            kwargs['partial'] = True
            instance = self.get_object()
            instance.deleted_at = None
            serializer = self.get_serializer(instance)
            message = status_message_response(
                            200, 'success',
                            'Archived service restored', serializer.data
                        )
            instance.save()
            return Response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # /archived - show list of archived services
    @action(methods=["GET"], detail=False)
    def archived(self, request, pk=None):
        try:
            queryset = self.queryset.filter(deleted_at__isnull=False)

            # table ordering
            if 'sort_order' in request.query_params and 'sort_field' in request.query_params:
                queryset = tbl_ordering(
                                    queryset, **request.query_params
                                )
            queryset = self.filter_queryset(queryset)

            if not queryset.exists():
                message = status_message_response(
                                200, 'success', 'No archived services', []
                            )
                return Response(message)

            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'Archived services found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # Show service endpoints
    @action(methods=['GET'], detail=True)
    def endpoints(self, request, pk):
        try:
            endpoints = APIEndpoint.objects.filter(service=pk)

            page = self.paginate_queryset(endpoints)
            if page is not None:
                serializer = APIEndpointSerializer(page, many=True)
                message = {
                    'code': 200,
                    'status': 'success',
                    'message': 'Service endpoints found',
                    'results': serializer.data
                }
                return self.get_paginated_response(message)

        except Exception as e:
            message = status_message_response(
                            500, 'failed',
                            'Request was not able to process' + str(e), []
                        )
            return Response(message,
                            status=status.HTTP_500_INTERNAL_SERVER_ERROR)
