import copy
import threading
from app.entities import enums
from django.db import transaction
from app.helper import decorators
from rest_framework.views import APIView
from app.helper.email_service import sender
from rest_framework import viewsets, status
from rest_framework.decorators import action
from django.contrib.auth import authenticate
from django.contrib.auth.hashers import check_password
from rest_framework.response import Response
from django.forms.models import model_to_dict
from app.entities.models import User, EntityLog, PasswordReset, Application
from app.helper.decorators import rms, error_safe
from django_filters import rest_framework as filters
from django.contrib.auth.hashers import make_password
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter
from app.applicationlayer.management.account import serializer
from app.applicationlayer.management.account.table_filters import AccountFilterset
from app.applicationlayer.utils import (
    CustomPagination, status_message_response, log_save,
    main_threading
)
from rest_framework.exceptions import ParseError


class UserViewSet(viewsets.ModelViewSet):
    queryset = User.objects.all().order_by('-id')
    serializer_class = serializer.UserSerializer
    pagination_class = CustomPagination
    lookup_field = 'code'
    filter_backends = (DjangoFilterBackend, SearchFilter, OrderingFilter)
    filterset_class = AccountFilterset
    ordering_fields = '__all__'
    search_fields = (
        'name', 'code',
        'department__company__name',
        'department__name',
        'email', 'contact_no'
    )

    @rms.user_create
    @transaction.atomic
    def create(self, request, *args, **kwargs):

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)

        password = User.objects.make_random_password(length=10)
        password_hash = make_password(password)

        app = request.data['application']

        if request.data['user_type'] != 'USR':
            rms = Application.objects.filter(id=1).values('code').first()
            app.append(rms['code'])
        # else:
        #     app = request.data['application']



        app = Application.objects.filter(
            code__in=app
        )

        instance = User.objects.get(
            id=serializer.data['id']
        )


        instance.application.set(app)

        User.objects.filter(
            id=serializer.data['id']
        ).update(password=password_hash)

        message = status_message_response(
            201, 'success',
            'New Users created', serializer.data
        )

        name = request.data['name']
        username = request.data['username']
        account_email = request.data['email']
        admin_email = request.user.email
        args = [name, username, password, account_email, admin_email]

        main_threading(args, sender.account_created)

        return Response(
            message,
            status=status.HTTP_201_CREATED
        )

    @decorators.rms.user_list
    def list(self, request, *args, **kwargs):

        queryset = self.filter_queryset(self.get_queryset())
        queryset = queryset.exclude(id=1)
        page = self.paginate_queryset(queryset)

        if page is not None:
            serializer = self.get_serializer(page, many=True)

            message = status_message_response(
                200,
                'success',
                'list of Users found',
                serializer.data
            )

            return self.get_paginated_response(message)

        serializer = self.get_serializer(queryset, many=True)

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )


    @decorators.rms.user_list
    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return Response(serializer.data)


    # @rms.user_create
    @error_safe
    @transaction.atomic
    def destroy(self, request, *args, **kwargs):

        instance = self.get_object()
        new_instance = model_to_dict(instance)
        self.perform_destroy(instance)

        log_save(
            enums.LogEnum.DELETED.value,
            enums.LogEntitiesEnum.USER.value,
            new_instance['id'],
            new_instance,
            ''
        )

        return Response(status=status.HTTP_204_NO_CONTENT)

    @rms.user_create
    @transaction.atomic
    def update(self, request, *args, **kwargs):

        partial = kwargs.pop('partial', False)
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data, partial=partial)
        serializer.is_valid(raise_exception=True)

        old_instance = model_to_dict(instance)
        self.perform_update(serializer)

        app = request.data['application']

        if request.data['user_type'] != 'USR':
            rms = Application.objects.filter(id=1).values('code').first()
            app.append(rms['code'])
        # else:
        #     app = request.data['application']



        app = Application.objects.filter(
            code__in=app
        )

        instance = User.objects.get(
            id=serializer.data['id']
        )


        instance.application.set(app)
        new_instance = serializer.data

        log_save(
            enums.LogEnum.UPDATE.value,
            enums.LogEntitiesEnum.USER.value,
            new_instance['id'],
            old_instance,
            new_instance
        )

        return Response(
            serializer.data,
            status=status.HTTP_200_OK
        )


    @action(detail=True,
            methods=['put'],
            url_path='reset-password',
            name="Reset Password of User")
    @decorators.error_safe
    @rms.user_create
    @transaction.atomic
    def ResetPassword(self, request, code=None):

        existingUser = User.objects.filter(code=str(code))
        pk = existingUser.values().first()['id']
        existingUser = existingUser.first()

        if existingUser:

            password = User.objects.make_random_password(length=10)
            existingUser.set_password(password)

            fromObj = copy.copy(existingUser)
            existingUser.save()
            toObj = copy.copy(existingUser)

            log_save(
                enums.LogEnum.UPDATE.value,
                enums.LogEntitiesEnum.USER.value,
                pk,
                model_to_dict(fromObj),
                model_to_dict(toObj))

            name = existingUser.name
            username = existingUser.username
            account_email = existingUser.email
            admin_email = request.user.email

            args = [name, username, password, account_email, admin_email]

            main_threading(args, sender.admin_changepassword)

        else:

            raise Exception('User not found')

        return Response(
            {"detail": "Success"},
            status=status.HTTP_200_OK
        )

    @action(detail=True,
            methods=['put'],
            url_path='change-password',
            name="Change Password of User")
    # @decorators.error_safe
    @transaction.atomic
    def ChangePassword(self, request, code=None):

        self.serializer_class = serializer.ChangePasswordSerializer
        serialized = self.get_serializer(data=request.data, context={'id': code})

        if serialized.is_valid():

            # form = copy.deepcopy(serialized.validated_data)
            form = request.data

            if form['new_password'] != form['new_password_confirm']:
                raise Exception('Passwords must match')

            existingUser = User.objects.filter(code=code)
            pk = existingUser.values().first()['id']


            if existingUser:

                existingUser.first().set_password(form['new_password_confirm'])
                fromObj = copy.copy(existingUser.first())
                existingUser.first().save()
                toObj = copy.copy(existingUser.first())

                log_save(
                    enums.LogEnum.UPDATE.value,
                    enums.LogEntitiesEnum.USER.value,
                    pk,
                    model_to_dict(fromObj),
                    model_to_dict(toObj)
                )

                return Response(
                    {"message": "Password successfully changed"},
                    status=status.HTTP_200_OK
                )
            else:
                raise Exception('User not found')
        else:
            serialized.is_valid(raise_exception=True)

        return Response(
            data={"detail": "Error"},
            status=status.HTTP_500_INTERNAL_SERVER_ERROR
        )
